/*
 * Decompiled with CFR 0.152.
 */
package net.straylightlabs.hola.sd;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.InetAddress;
import java.net.MulticastSocket;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import net.straylightlabs.hola.dns.ARecord;
import net.straylightlabs.hola.dns.AaaaRecord;
import net.straylightlabs.hola.dns.Domain;
import net.straylightlabs.hola.dns.PtrRecord;
import net.straylightlabs.hola.dns.Question;
import net.straylightlabs.hola.dns.Record;
import net.straylightlabs.hola.dns.Response;
import net.straylightlabs.hola.dns.SrvRecord;
import net.straylightlabs.hola.dns.TxtRecord;
import net.straylightlabs.hola.sd.Instance;
import net.straylightlabs.hola.sd.Service;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Query {
    private final Service service;
    private final Domain domain;
    private final int browsingTimeout;
    private final Lock socketLock;
    private MulticastSocket socket;
    private InetAddress mdnsGroupIPv4;
    private InetAddress mdnsGroupIPv6;
    private boolean isUsingIPv4;
    private boolean isUsingIPv6;
    private Question initialQuestion;
    private Set<Question> questions;
    private Set<Instance> instances;
    private Set<Record> records;
    private boolean listenerStarted;
    private boolean listenerFinished;
    private static final Logger logger = LoggerFactory.getLogger(Query.class);
    public static final String MDNS_IP4_ADDRESS = "224.0.0.251";
    public static final String MDNS_IP6_ADDRESS = "FF02::FB";
    public static final int MDNS_PORT = 5353;
    private static final int WAIT_FOR_LISTENER_MS = 10;
    private static final int BROWSING_TIMEOUT = 750;

    public static Query createFor(Service service, Domain domain) {
        return new Query(service, domain, 750);
    }

    public static Query createWithTimeout(Service service, Domain domain, int timeout) {
        return new Query(service, domain, timeout);
    }

    private Query(Service service, Domain domain, int browsingTimeout) {
        this.service = service;
        this.domain = domain;
        this.browsingTimeout = browsingTimeout;
        this.questions = new HashSet<Question>();
        this.records = new HashSet<Record>();
        this.socketLock = new ReentrantLock();
    }

    public Set<Instance> runOnce() throws IOException {
        this.initialQuestion = new Question(this.service, this.domain);
        this.instances = Collections.synchronizedSet(new HashSet());
        try {
            this.openSocket();
            Thread listener = this.listenForResponses();
            while (!this.isServerIsListening()) {
                logger.debug("Server is not yet listening");
            }
            this.ask(this.initialQuestion);
            try {
                listener.join();
            }
            catch (InterruptedException e) {
                logger.error("InterruptedException while listening for mDNS responses: ", (Throwable)e);
            }
        }
        finally {
            this.closeSocket();
        }
        return this.instances;
    }

    private void ask(Question question) throws IOException {
        if (this.questions.contains(question)) {
            logger.debug("We've already asked {}, we won't ask again", (Object)question);
            return;
        }
        this.questions.add(question);
        if (this.isUsingIPv4) {
            question.askOn(this.socket, this.mdnsGroupIPv4);
        }
        if (this.isUsingIPv6) {
            question.askOn(this.socket, this.mdnsGroupIPv6);
        }
    }

    private boolean isServerIsListening() {
        boolean retval;
        try {
            while (!this.socketLock.tryLock(10L, TimeUnit.MILLISECONDS)) {
                this.socketLock.notify();
                logger.debug("Waiting to acquire socket lock");
            }
            if (this.listenerFinished) {
                throw new RuntimeException("Listener has already finished");
            }
            retval = this.listenerStarted;
        }
        catch (InterruptedException e) {
            logger.error("Interrupted while waiting to acquire socket lock: ", (Throwable)e);
            throw new RuntimeException("Server is not listening");
        }
        finally {
            this.socketLock.unlock();
        }
        return retval;
    }

    public void start() {
        throw new RuntimeException("Not implemented yet");
    }

    private void openSocket() throws IOException {
        this.mdnsGroupIPv4 = InetAddress.getByName(MDNS_IP4_ADDRESS);
        this.mdnsGroupIPv6 = InetAddress.getByName(MDNS_IP6_ADDRESS);
        this.socket = new MulticastSocket(5353);
        try {
            this.socket.joinGroup(this.mdnsGroupIPv4);
            this.isUsingIPv4 = true;
        }
        catch (SocketException e) {
            logger.error("SocketException when joining group for {}, IPv4-only hosts will not be found", (Object)MDNS_IP4_ADDRESS, (Object)e);
        }
        try {
            this.socket.joinGroup(this.mdnsGroupIPv6);
            this.isUsingIPv6 = true;
        }
        catch (SocketException e) {
            logger.error("SocketException when joining group for {}, IPv6-only hosts will not be found", (Object)MDNS_IP6_ADDRESS, (Object)e);
        }
        if (!this.isUsingIPv4 && !this.isUsingIPv6) {
            throw new IOException("No usable network interfaces found");
        }
        this.socket.setTimeToLive(10);
        this.socket.setSoTimeout(this.browsingTimeout);
    }

    private Thread listenForResponses() {
        Thread listener = new Thread(this::collectResponses);
        listener.start();
        return listener;
    }

    private Set<Instance> collectResponses() {
        long startTime;
        long currentTime = startTime = System.currentTimeMillis();
        this.socketLock.lock();
        this.listenerStarted = true;
        this.listenerFinished = false;
        this.socketLock.unlock();
        int timeouts = 0;
        while (timeouts == 0 && currentTime - startTime < (long)this.browsingTimeout) {
            byte[] responseBuffer = new byte[9000];
            DatagramPacket responsePacket = new DatagramPacket(responseBuffer, responseBuffer.length);
            try {
                logger.debug("Listening for responses...");
                this.socket.receive(responsePacket);
                currentTime = System.currentTimeMillis();
                logger.debug("Response received!");
                try {
                    Response response = Response.createFrom(responsePacket);
                    if (!response.answers(this.questions)) {
                        logger.debug("This response doesn't answer any of our questions, ignoring it.");
                        timeouts = 0;
                        continue;
                    }
                    this.records.addAll(response.getRecords());
                    this.fetchMissingRecords();
                }
                catch (IllegalArgumentException e) {
                    logger.debug("Response was not a mDNS response packet, ignoring it");
                    timeouts = 0;
                    continue;
                }
                timeouts = 0;
            }
            catch (SocketTimeoutException e) {
                ++timeouts;
            }
            catch (IOException e) {
                logger.error("IOException while listening for mDNS responses: ", (Throwable)e);
            }
        }
        this.socketLock.lock();
        this.listenerFinished = true;
        this.socketLock.unlock();
        this.buildInstancesFromRecords();
        return this.instances;
    }

    private void fetchMissingRecords() throws IOException {
        logger.debug("Records includes:");
        this.records.stream().forEach(r -> logger.debug("{}", r));
        for (PtrRecord ptr : this.records.stream().filter(r -> r instanceof PtrRecord).map(r -> (PtrRecord)r).collect(Collectors.toList())) {
            this.fetchMissingSrvRecordsFor(ptr);
            this.fetchMissingTxtRecordsFor(ptr);
        }
        for (SrvRecord srv : this.records.stream().filter(r -> r instanceof SrvRecord).map(r -> (SrvRecord)r).collect(Collectors.toList())) {
            this.fetchMissingAddressRecordsFor(srv);
        }
    }

    private void fetchMissingSrvRecordsFor(PtrRecord ptr) throws IOException {
        long numRecords = this.records.stream().filter(r -> r instanceof SrvRecord).filter(r -> r.getName().equals(ptr.getPtrName())).count();
        if (numRecords == 0L) {
            logger.debug("Response has no SRV records");
            this.querySrvRecordFor(ptr);
        }
    }

    private void fetchMissingTxtRecordsFor(PtrRecord ptr) throws IOException {
        long numRecords = this.records.stream().filter(r -> r instanceof TxtRecord).filter(r -> r.getName().equals(ptr.getPtrName())).count();
        if (numRecords == 0L) {
            logger.debug("Response has no TXT records");
            this.queryTxtRecordFor(ptr);
        }
    }

    private void fetchMissingAddressRecordsFor(SrvRecord srv) throws IOException {
        long numRecords = this.records.stream().filter(r -> r instanceof ARecord || r instanceof AaaaRecord).filter(r -> r.getName().equals(srv.getTarget())).count();
        if (numRecords == 0L) {
            logger.debug("Response has no A or AAAA records");
            this.queryAddressesFor(srv);
        }
    }

    private void querySrvRecordFor(PtrRecord ptr) throws IOException {
        Question question = new Question(ptr.getPtrName(), Question.QType.SRV, Question.QClass.IN);
        this.ask(question);
    }

    private void queryTxtRecordFor(PtrRecord ptr) throws IOException {
        Question question = new Question(ptr.getPtrName(), Question.QType.TXT, Question.QClass.IN);
        this.ask(question);
    }

    private void queryAddressesFor(SrvRecord srv) throws IOException {
        Question question = new Question(srv.getTarget(), Question.QType.A, Question.QClass.IN);
        this.ask(question);
        question = new Question(srv.getTarget(), Question.QType.AAAA, Question.QClass.IN);
        this.ask(question);
    }

    private void buildInstancesFromRecords() {
        this.records.stream().filter(r -> r instanceof PtrRecord && this.initialQuestion.answeredBy((Record)r)).map(r -> (PtrRecord)r).forEach(ptr -> this.instances.add(Instance.createFromRecords(ptr, this.records)));
    }

    private void closeSocket() {
        if (this.socket != null) {
            this.socket.close();
            this.socket = null;
        }
    }
}

