package net.thucydides.core.reports.adaptors;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import net.thucydides.core.guice.Injectors;
import net.thucydides.core.reports.adaptors.lettuce.LettuceXUnitAdaptor;
import net.thucydides.core.reports.adaptors.xunit.DefaultXUnitAdaptor;
import net.thucydides.core.util.EnvironmentVariables;
import org.apache.commons.lang3.StringUtils;

import java.util.Map;

public class AdaptorService {

    private final static Map<String, ? extends TestOutcomeAdaptor> BUILT_IN_ADAPTORS
            = ImmutableMap.of("xunit", new DefaultXUnitAdaptor(),
                              "lettuce", new LettuceXUnitAdaptor());

    private final EnvironmentVariables environmentVariables;

    public AdaptorService(EnvironmentVariables environmentVariables) {
        this.environmentVariables = environmentVariables;
    }

    public AdaptorService() {
        this(Injectors.getInjector().getInstance(EnvironmentVariables.class));
    }

    public TestOutcomeAdaptor getAdaptor(String name) {
        if (BUILT_IN_ADAPTORS.containsKey(name)) {
            return BUILT_IN_ADAPTORS.get(name);
        }
        String customAdaptor = environmentVariables.getProperty("thucydides.xunit.adaptors." + name);
        if (StringUtils.isNotEmpty(customAdaptor)) {
            return newAdaptor(customAdaptor);
        }
        throw new UnknownAdaptor("Unknown test outcome adaptor: " + name);
    }

    private TestOutcomeAdaptor newAdaptor(String customAdaptor) {
        try {
            return (TestOutcomeAdaptor) Class.forName(customAdaptor).newInstance();
        } catch (Exception e) {
            throw new UnknownAdaptor("Test outcome adaptor could not be instanciated: " + customAdaptor);
        }
    }
}
