/*
 * Decompiled with CFR 0.152.
 */
package net.time4j.history;

import java.io.IOException;
import java.io.InvalidObjectException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import net.time4j.CalendarUnit;
import net.time4j.PlainDate;
import net.time4j.engine.ChronoElement;
import net.time4j.engine.EpochDays;
import net.time4j.engine.TimePoint;
import net.time4j.format.TextElement;
import net.time4j.history.AncientJulianLeapYears;
import net.time4j.history.Calculus;
import net.time4j.history.CalendarAlgorithm;
import net.time4j.history.CutOverEvent;
import net.time4j.history.HistoricDate;
import net.time4j.history.HistoricEra;
import net.time4j.history.HistoricalEraElement;
import net.time4j.history.HistoricalIntegerElement;
import net.time4j.history.SPX;
import net.time4j.history.internal.HistoricVariant;

public final class ChronoHistory
implements Serializable {
    public static final ChronoHistory PROLEPTIC_GREGORIAN = new ChronoHistory(HistoricVariant.PROLEPTIC_GREGORIAN, Collections.singletonList(new CutOverEvent(Long.MIN_VALUE, CalendarAlgorithm.GREGORIAN, CalendarAlgorithm.GREGORIAN)));
    public static final ChronoHistory PROLEPTIC_JULIAN = new ChronoHistory(HistoricVariant.PROLEPTIC_JULIAN, Collections.singletonList(new CutOverEvent(Long.MIN_VALUE, CalendarAlgorithm.JULIAN, CalendarAlgorithm.JULIAN)));
    private static final long EARLIEST_CUTOVER = (Long)PlainDate.of((int)1582, (int)10, (int)15).get((ChronoElement)EpochDays.MODIFIED_JULIAN_DATE);
    private static final ChronoHistory INTRODUCTION_BY_POPE_GREGOR = ChronoHistory.ofGregorianReform(EARLIEST_CUTOVER);
    private static final ChronoHistory SWEDEN;
    private static final Map<String, ChronoHistory> LOOKUP;
    private static final long serialVersionUID = 4100690610730913643L;
    private final transient HistoricVariant variant;
    private final transient List<CutOverEvent> events;
    private final transient AncientJulianLeapYears ajly;
    private final transient ChronoElement<HistoricEra> eraElement;
    private final transient TextElement<Integer> yearOfEraElement;
    private final transient TextElement<Integer> monthElement;
    private final transient TextElement<Integer> dayOfMonthElement;
    private final transient Set<ChronoElement<?>> elements;

    private ChronoHistory(HistoricVariant historicVariant, List<CutOverEvent> list) {
        this(historicVariant, list, null);
    }

    private ChronoHistory(HistoricVariant historicVariant, List<CutOverEvent> list, AncientJulianLeapYears ancientJulianLeapYears) {
        if (list.isEmpty()) {
            throw new IllegalArgumentException("At least one cutover event must be present in chronological history.");
        }
        if (historicVariant == null) {
            throw new NullPointerException("Missing historical variant.");
        }
        this.variant = historicVariant;
        this.events = list;
        this.ajly = ancientJulianLeapYears;
        this.eraElement = new HistoricalEraElement(this);
        this.yearOfEraElement = HistoricalIntegerElement.forYearOfEra(this);
        this.monthElement = HistoricalIntegerElement.forMonth(this);
        this.dayOfMonthElement = HistoricalIntegerElement.forDayOfMonth(this);
        HashSet<Object> hashSet = new HashSet<Object>();
        hashSet.add(this.eraElement);
        hashSet.add(this.yearOfEraElement);
        hashSet.add(this.monthElement);
        hashSet.add(this.dayOfMonthElement);
        this.elements = Collections.unmodifiableSet(hashSet);
    }

    public static ChronoHistory ofFirstGregorianReform() {
        return INTRODUCTION_BY_POPE_GREGOR;
    }

    public static ChronoHistory ofGregorianReform(PlainDate plainDate) {
        if (plainDate.equals((Object)PlainDate.axis().getMaximum())) {
            return PROLEPTIC_JULIAN;
        }
        if (plainDate.equals((Object)PlainDate.axis().getMinimum())) {
            return PROLEPTIC_GREGORIAN;
        }
        long l = (Long)plainDate.get((ChronoElement)EpochDays.MODIFIED_JULIAN_DATE);
        ChronoHistory.check(l);
        if (l == EARLIEST_CUTOVER) {
            return INTRODUCTION_BY_POPE_GREGOR;
        }
        return ChronoHistory.ofGregorianReform(l);
    }

    public static ChronoHistory of(Locale locale) {
        ChronoHistory chronoHistory = LOOKUP.get(locale.getCountry());
        if (chronoHistory == null) {
            return ChronoHistory.ofFirstGregorianReform();
        }
        return chronoHistory;
    }

    public static ChronoHistory ofSweden() {
        return SWEDEN;
    }

    public boolean isValid(HistoricDate historicDate) {
        if (historicDate == null) {
            return false;
        }
        Calculus calculus = this.getAlgorithm(historicDate);
        return calculus != null && calculus.isValid(historicDate);
    }

    public PlainDate convert(HistoricDate historicDate) {
        Calculus calculus = this.getAlgorithm(historicDate);
        if (calculus == null) {
            throw new IllegalArgumentException("Invalid historical date: " + historicDate);
        }
        return PlainDate.of((long)calculus.toMJD(historicDate), (EpochDays)EpochDays.MODIFIED_JULIAN_DATE);
    }

    public HistoricDate convert(PlainDate plainDate) {
        long l = (Long)plainDate.get((ChronoElement)EpochDays.MODIFIED_JULIAN_DATE);
        for (int i = this.events.size() - 1; i >= 0; --i) {
            CutOverEvent cutOverEvent = this.events.get(i);
            if (l < cutOverEvent.start) continue;
            return cutOverEvent.algorithm.fromMJD(l);
        }
        return this.getJulianAlgorithm().fromMJD(l);
    }

    public String getVariant() {
        StringBuilder stringBuilder = new StringBuilder(64);
        stringBuilder.append(this.variant.name());
        switch (this.variant) {
            case PROLEPTIC_GREGORIAN: 
            case PROLEPTIC_JULIAN: {
                stringBuilder.append(":no-cutover");
                break;
            }
            case INTRODUCTION_ON_1582_10_15: 
            case SINGLE_CUTOVER_DATE: {
                stringBuilder.append(":cutover=");
                stringBuilder.append(this.getGregorianCutOverDate());
            }
            default: {
                stringBuilder.append(":ancient-julian-leap-years=");
                if (this.ajly != null) {
                    int[] nArray = this.ajly.getPattern();
                    stringBuilder.append('[');
                    stringBuilder.append(nArray[0]);
                    for (int i = 1; i < nArray.length; ++i) {
                        stringBuilder.append(',');
                        stringBuilder.append(nArray[i]);
                    }
                    stringBuilder.append(']');
                    break;
                }
                stringBuilder.append("[]");
            }
        }
        return stringBuilder.toString();
    }

    public PlainDate getGregorianCutOverDate() {
        long l = this.events.get((int)(this.events.size() - 1)).start;
        if (l == Long.MIN_VALUE) {
            throw new UnsupportedOperationException("Proleptic history without any gregorian reform date.");
        }
        return PlainDate.of((long)l, (EpochDays)EpochDays.MODIFIED_JULIAN_DATE);
    }

    public boolean hasGregorianCutOverDate() {
        return this.events.get((int)(this.events.size() - 1)).start > Long.MIN_VALUE;
    }

    public int getLengthOfYear(HistoricEra historicEra, int n) {
        try {
            HistoricDate historicDate = HistoricDate.of(historicEra, n, 1, 1);
            HistoricDate historicDate2 = HistoricDate.of(historicEra, n, 12, 31);
            return (int)(CalendarUnit.DAYS.between((TimePoint)this.convert(historicDate), (TimePoint)this.convert(historicDate2)) + 1L);
        }
        catch (RuntimeException runtimeException) {
            return -1;
        }
    }

    public AncientJulianLeapYears getAncientJulianLeapYears() {
        if (this.ajly == null) {
            throw new UnsupportedOperationException("No historic julian leap years were defined.");
        }
        return this.ajly;
    }

    public boolean hasAncientJulianLeapYears() {
        return this.ajly != null;
    }

    public ChronoHistory with(AncientJulianLeapYears ancientJulianLeapYears) {
        if (ancientJulianLeapYears == null) {
            throw new NullPointerException("Missing ancient julian leap years.");
        }
        if (!this.hasGregorianCutOverDate()) {
            return this;
        }
        return new ChronoHistory(this.variant, this.events, ancientJulianLeapYears);
    }

    public ChronoElement<HistoricEra> era() {
        return this.eraElement;
    }

    public TextElement<Integer> yearOfEra() {
        return this.yearOfEraElement;
    }

    public TextElement<Integer> month() {
        return this.monthElement;
    }

    public TextElement<Integer> dayOfMonth() {
        return this.dayOfMonthElement;
    }

    public Set<ChronoElement<?>> getElements() {
        return this.elements;
    }

    public boolean equals(Object object) {
        if (this == object) {
            return true;
        }
        if (object instanceof ChronoHistory) {
            ChronoHistory chronoHistory = (ChronoHistory)object;
            if (this.variant == chronoHistory.variant && ChronoHistory.isEqual(this.ajly, chronoHistory.ajly)) {
                return this.variant != HistoricVariant.SINGLE_CUTOVER_DATE || this.events.get((int)0).start == chronoHistory.events.get((int)0).start;
            }
        }
        return false;
    }

    public int hashCode() {
        if (this.variant == HistoricVariant.SINGLE_CUTOVER_DATE) {
            long l = this.events.get((int)0).start;
            return (int)(l ^ l << 32);
        }
        return this.variant.hashCode();
    }

    public String toString() {
        String string;
        switch (this.variant) {
            case PROLEPTIC_GREGORIAN: {
                string = "ChronoHistory[PROLEPTIC-GREGORIAN]";
                break;
            }
            case PROLEPTIC_JULIAN: {
                string = "ChronoHistory[PROLEPTIC-JULIAN]";
                break;
            }
            case SWEDEN: {
                string = "ChronoHistory[SWEDEN]";
                break;
            }
            default: {
                PlainDate plainDate = this.getGregorianCutOverDate();
                string = "ChronoHistory[" + plainDate + "]";
            }
        }
        if (this.ajly != null) {
            string = string + " with ancient julian leap years: " + this.ajly;
        }
        return string;
    }

    Calculus getAlgorithm(HistoricDate historicDate) {
        for (int i = this.events.size() - 1; i >= 0; --i) {
            CutOverEvent cutOverEvent = this.events.get(i);
            if (historicDate.compareTo(cutOverEvent.dateAtCutOver) >= 0) {
                return cutOverEvent.algorithm;
            }
            if (historicDate.compareTo(cutOverEvent.dateBeforeCutOver) <= 0) continue;
            return null;
        }
        return this.getJulianAlgorithm();
    }

    HistoricDate adjustDayOfMonth(HistoricDate historicDate) {
        Calculus calculus = this.getAlgorithm(historicDate);
        if (calculus == null) {
            return historicDate;
        }
        int n = calculus.getMaximumDayOfMonth(historicDate);
        if (n < historicDate.getDayOfMonth()) {
            return HistoricDate.of(historicDate.getEra(), historicDate.getYearOfEra(), historicDate.getMonth(), n);
        }
        return historicDate;
    }

    List<CutOverEvent> getEvents() {
        return this.events;
    }

    HistoricVariant getHistoricVariant() {
        return this.variant;
    }

    private Calculus getJulianAlgorithm() {
        if (this.ajly != null) {
            return this.ajly.getCalculus();
        }
        return CalendarAlgorithm.JULIAN;
    }

    private static boolean isEqual(Object object, Object object2) {
        return object == null ? object2 == null : object.equals(object2);
    }

    private static void check(long l) {
        if (l < EARLIEST_CUTOVER) {
            throw new IllegalArgumentException("Gregorian calendar did not exist before 1582-10-15");
        }
    }

    private static ChronoHistory ofGregorianReform(long l) {
        return new ChronoHistory(l == EARLIEST_CUTOVER ? HistoricVariant.INTRODUCTION_ON_1582_10_15 : HistoricVariant.SINGLE_CUTOVER_DATE, Collections.singletonList(new CutOverEvent(l, CalendarAlgorithm.JULIAN, CalendarAlgorithm.GREGORIAN)));
    }

    private Object writeReplace() {
        return new SPX(this, 2);
    }

    private void readObject(ObjectInputStream objectInputStream) throws IOException {
        throw new InvalidObjectException("Serialization proxy required.");
    }

    static {
        ArrayList<CutOverEvent> arrayList = new ArrayList<CutOverEvent>();
        arrayList.add(new CutOverEvent(-57959L, CalendarAlgorithm.JULIAN, CalendarAlgorithm.SWEDISH));
        arrayList.add(new CutOverEvent(-53575L, CalendarAlgorithm.SWEDISH, CalendarAlgorithm.JULIAN));
        arrayList.add(new CutOverEvent(-38611L, CalendarAlgorithm.JULIAN, CalendarAlgorithm.GREGORIAN));
        SWEDEN = new ChronoHistory(HistoricVariant.SWEDEN, Collections.unmodifiableList(arrayList));
        HashMap<String, ChronoHistory> hashMap = new HashMap<String, ChronoHistory>();
        hashMap.put("GB", ChronoHistory.ofGregorianReform(PlainDate.of((int)1752, (int)9, (int)14)));
        hashMap.put("RU", ChronoHistory.ofGregorianReform(PlainDate.of((int)1918, (int)2, (int)14)));
        hashMap.put("SE", SWEDEN);
        LOOKUP = Collections.unmodifiableMap(hashMap);
    }
}

