/*
 * Copyright (c) 2016, Peter Rader. All rights reserved.
 *  ___ ___               __                 ______         __     __  __         __
 * |   |   |.-----..----.|  |_ .-----..----.|   __ \.--.--.|  |--.|  ||__|.-----.|  |--.
 * |   |   ||  -__||  __||   _||  _  ||   _||    __/|  |  ||  _  ||  ||  ||__ --||     |
 *  \_____/ |_____||____||____||_____||__|  |___|   |_____||_____||__||__||_____||__|__|
 *
 * http://www.gnu.org/licenses/gpl-3.0.html
 */
package net.vectorpublish.desktop.vp;

import java.awt.Rectangle;
import java.io.File;
import java.io.IOException;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.logging.ConsoleHandler;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.inject.Named;

import net.vectorpublish.desktop.vp.api.conf.Config;
import net.vectorpublish.desktop.vp.conf.FileProperties;
import net.vectorpublish.desktop.vp.ui.Namespace;

@Named
public class ConfigImpl implements Config {

	private static final String SEPARATOR = ".";

	private static final Set<Logger> avoidGarbagecollectConfiguredLevels = new LinkedHashSet<>();
	private FileProperties propertiesFile;

	@Override
	public Rectangle loadBounds(Namespace namespace, String key) {
		final String name = namespace + SEPARATOR + key;
		try {
			final int x = Integer.parseInt(propertiesFile.get(name + ".x") + "");
			final int y = Integer.parseInt(propertiesFile.get(name + ".y") + "");
			final int w = Integer.parseInt(propertiesFile.get(name + ".w") + "");
			final int h = Integer.parseInt(propertiesFile.get(name + ".h") + "");
			return new Rectangle(x, y, w, h);
		} catch (final NumberFormatException e) {
			return null;
		}
	}

	@PostConstruct
	private final void read() throws IOException {
		propertiesFile = new FileProperties(new File("config.properties"));
		for (Object key : propertiesFile.keySet()) {
			final Object value = propertiesFile.get(key);
			if (key.toString().startsWith("log.")) {
				key = key.toString().substring(4);
			} else {
				continue;
			}
			Level level;

			try {
				level = Level.parse("" + value);
			} catch (final IllegalArgumentException e) {
				level = null;
			}
			Class target;
			try {
				target = Class.forName("" + key);
			} catch (final ClassNotFoundException e) {
				target = null;
			}
			if (level == null && target != null) {
				System.err.println(value + " is not a valid JUL logging level for " + value);
			} else {
				final Logger logger = Logger.getLogger(key + "");
				if (level != null && target == null) {
					System.err.println(key + " class not found for logging on level " + level + "!");
					System.err.println("Try to activate logger for category " + key + " on level " + level + "!");
				} else if (level != null && target != null) {
					System.err.println("Activate JUL logging on level " + level + " for class " + target);
				}
				logger.setLevel(level);
				final ConsoleHandler handler = new ConsoleHandler();
				handler.setLevel(level);
				logger.addHandler(handler);
				avoidGarbagecollectConfiguredLevels.add(logger);
			}
		}
	}

	@Override
	public String read(Namespace namespace, String key) {
		return propertiesFile.getProperty(namespace.getConfigNamespace() + SEPARATOR + key);
	}

	@Override
	public void storeBounds(Namespace namespace, String key, Rectangle bounds) {
		final String name = namespace + SEPARATOR + key;
		propertiesFile.setProperty(name + ".x", "" + bounds.x);
		propertiesFile.setProperty(name + ".y", "" + bounds.y);
		propertiesFile.setProperty(name + ".w", "" + bounds.width);
		propertiesFile.setProperty(name + ".h", "" + bounds.height);

	}

	@PreDestroy
	private final void write() throws IOException {
		propertiesFile.store();
	}

	@Override
	public void write(Namespace namespace, String key, String value) {
		propertiesFile.setProperty(namespace.getConfigNamespace() + SEPARATOR + key, value);
	}

}
