/*
 * Copyright (c) 2016, Peter Rader. All rights reserved.
 *  ___ ___               __                 ______         __     __  __         __
 * |   |   |.-----..----.|  |_ .-----..----.|   __ \.--.--.|  |--.|  ||__|.-----.|  |--.
 * |   |   ||  -__||  __||   _||  _  ||   _||    __/|  |  ||  _  ||  ||  ||__ --||     |
 *  \_____/ |_____||____||____||_____||__|  |___|   |_____||_____||__||__||_____||__|__|
 *
 * http://www.gnu.org/licenses/gpl-3.0.html
 */
package net.vectorpublish.desktop.vp;

import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.Serializable;

import javax.inject.Inject;
import javax.inject.Named;

import net.vectorpublish.desktop.vp.api.history.HistoryGrownListener;
import net.vectorpublish.desktop.vp.api.history.ReadOnlyHistoryStepDataBean;
import net.vectorpublish.desktop.vp.api.ui.DocumentSelectionChangeListener;
import net.vectorpublish.desktop.vp.api.vpd.DocumentNode;
import net.vectorpublish.desktop.vp.log.Log;

@Named
public class HistoryImpl extends History implements DocumentSelectionChangeListener {

	public class HistoryWriter {

		private final ObjectOutputStream os;

		public HistoryWriter(ObjectOutputStream os) {
			this.os = os;
		}

		/**
		 * Writes all {@link HistoryStep steps} to the {@link #os stream}.
		 *
		 * @param doc
		 *            The {@link DocumentNode document}.
		 * @throws IOException
		 *             When the stream could not be written or the
		 *             {@link ReadOnlyHistoryStepDataBean bean} is not
		 *             {@link Serializable}.
		 */
		public void writeHistorySteps(DocumentNode doc) throws IOException {
			HistoryStep<?> step = doc.getLastExecutedHistoryStep();
			int currentHistoryIndex = 0;
			while (step.previous != null) {
				currentHistoryIndex++;
				step = step.previous;
			}
			os.writeInt(currentHistoryIndex);
			// Step points to the first now.
			while (step != null) {
				os.writeBoolean(true);
				os.writeUTF(step.getClass().getName());
				log.system("Write " + step.data);
				os.writeObject(step.data);
				step = step.next;
			}
			os.writeBoolean(false);
		}

	}

	@Inject
	private final Log log = null;

	private DocumentNode currentDocument = null;

	@Inject
	private final HistoryGrownListener[] grownListener = null;

	@Override
	public DocumentNode getCurrentDocument() {
		return currentDocument;
	}

	@Override
	public void notifyDocumentSelected(DocumentNode newCurrentDocument) {
		this.currentDocument = newCurrentDocument;
	}

	@Override
	protected void notifyHistoryChanged(HistoryStep<?> previous, HistoryStep<?> newStep, boolean byFileRead) {
		if (previous != null) {
			for (final HistoryGrownListener listener : grownListener) {
				listener.notifyHistoryGrown(newStep, byFileRead);
			}
		}
	}
}
