/*
 * Copyright (c) 2016, Peter Rader. All rights reserved.
 *  ___ ___               __                 ______         __     __  __         __
 * |   |   |.-----..----.|  |_ .-----..----.|   __ \.--.--.|  |--.|  ||__|.-----.|  |--.
 * |   |   ||  -__||  __||   _||  _  ||   _||    __/|  |  ||  _  ||  ||  ||__ --||     |
 *  \_____/ |_____||____||____||_____||__|  |___|   |_____||_____||__||__||_____||__|__|
 *
 * http://www.gnu.org/licenses/gpl-3.0.html
 */
package net.vectorpublish.desktop.vp;

import java.awt.event.ActionEvent;
import java.util.Set;

import javax.annotation.PostConstruct;
import javax.inject.Inject;
import javax.inject.Named;

import org.springframework.beans.factory.SmartInitializingSingleton;
import org.springframework.beans.factory.config.AutowireCapableBeanFactory;
import org.springframework.context.annotation.DependsOn;

import net.vectorpublish.desktop.vp.History.HistoryStep;
import net.vectorpublish.desktop.vp.api.EditMenu;
import net.vectorpublish.desktop.vp.api.history.BackwardHistoryStepListener;
import net.vectorpublish.desktop.vp.api.history.ForwardHistoryStepListener;
import net.vectorpublish.desktop.vp.api.history.HistoryGrownListener;
import net.vectorpublish.desktop.vp.api.history.Redo;
import net.vectorpublish.desktop.vp.api.history.Undo;
import net.vectorpublish.desktop.vp.api.ui.DocumentSelectionChangeListener;
import net.vectorpublish.desktop.vp.api.ui.ToolBar;
import net.vectorpublish.desktop.vp.api.ui.VPAbstractAction;
import net.vectorpublish.desktop.vp.api.ui.kf.I8nTextKeyframe;
import net.vectorpublish.desktop.vp.api.vpd.DocumentNode;
import net.vectorpublish.desktop.vp.i8n.I8n;
import net.vectorpublish.desktop.vp.log.Log;
import net.vectorpublish.desktop.vp.ui.ImageKey;
import net.vectorpublish.desktop.vp.ui.Namespace;

@Named
@DependsOn("undoAction")
public class RedoAction extends VPAbstractAction implements DocumentSelectionChangeListener, HistoryGrownListener,
		ForwardHistoryStepListener, BackwardHistoryStepListener, Redo, SmartInitializingSingleton {

	public final static Namespace NS = Namespace.getNamespace("net.vectorpublish", "history");

	/**
	 * The toolbar to hold the redo button.
	 */
	@Inject
	private final EditMenu edit = null;

	@Inject
	private final ToolBar toolbar = null;

	@Inject
	private final AutowireCapableBeanFactory bf = null;

	private DocumentNode doc;

	@Inject
	private final Set<ForwardHistoryStepListener> listeners = null;

	@Inject
	private final Undo undo = null;

	@Inject
	private final Log log = null;

	/**
	 * Injects a CDI-Bean.
	 */
	@Inject
	private final I8n i8n = null;

	public RedoAction() {
		super(I8nTextKeyframe.REDO, I8nTextKeyframe.REDO_DESC, true);
	}

	@Override
	public void actionPerformed(ActionEvent e) {
		log.methodIn("Redo");
		final HistoryStep<?> currentHistoryStep = doc.getLastExecutedHistoryStep();
		currentHistoryStep.executeNext();
		for (final ForwardHistoryStepListener listener : listeners) {
			listener.notifyHSDone();
		}
		// TODO: remove next line when https://jira.spring.io/browse/SPR-8450
		// fixed.
		if (!listeners.contains(this)) {
			notifyHSDone();
		}
	}

	@Override
	public void afterSingletonsInstantiated() {
		toolbar.moveToTheRight(undo, this);
	}

	@Override
	public void notifyDocumentSelected(DocumentNode selectedDocument) {
		this.doc = selectedDocument;
		updateState();
	}

	@Override
	public void notifyHistoryGrown(HistoryStep<?> incomming, boolean byFileRead) {
		if (!byFileRead) {
			updateState();
		}
	}

	@Override
	public void notifyHSDone() {
		updateState();
	}

	@Override
	public void notifyHSUndone() {
		updateState();
	}

	@PostConstruct
	private final void setup() {
		setIcons(NS, ImageKey.get("redo"));
		edit.add(this);
		toolbar.add(this);

	}

	private void updateState() {
		if (doc == null) {
			setEnabled(false);
			return;
		}
		final HistoryStep<?> lastExecutedHistoryStep = doc.getLastExecutedHistoryStep();
		setEnabled(lastExecutedHistoryStep.canExecuteNext());
	}
}
