/*
 * Copyright (c) 2016, Peter Rader. All rights reserved.
 *  ___ ___               __                 ______         __     __  __         __
 * |   |   |.-----..----.|  |_ .-----..----.|   __ \.--.--.|  |--.|  ||__|.-----.|  |--.
 * |   |   ||  -__||  __||   _||  _  ||   _||    __/|  |  ||  _  ||  ||  ||__ --||     |
 *  \_____/ |_____||____||____||_____||__|  |___|   |_____||_____||__||__||_____||__|__|
 *
 * http://www.gnu.org/licenses/gpl-3.0.html
 */
package net.vectorpublish.desktop.vp;

import java.awt.event.ActionEvent;
import java.util.Set;

import javax.annotation.PostConstruct;
import javax.inject.Inject;
import javax.inject.Named;

import net.vectorpublish.desktop.vp.History.HistoryStep;
import net.vectorpublish.desktop.vp.api.EditMenu;
import net.vectorpublish.desktop.vp.api.history.BackwardHistoryStepListener;
import net.vectorpublish.desktop.vp.api.history.ForwardHistoryStepListener;
import net.vectorpublish.desktop.vp.api.history.HistoryGrownListener;
import net.vectorpublish.desktop.vp.api.history.Undo;
import net.vectorpublish.desktop.vp.api.ui.DocumentSelectionChangeListener;
import net.vectorpublish.desktop.vp.api.ui.ToolBar;
import net.vectorpublish.desktop.vp.api.ui.VPAbstractAction;
import net.vectorpublish.desktop.vp.api.ui.kf.I8nTextKeyframe;
import net.vectorpublish.desktop.vp.api.vpd.DocumentNode;
import net.vectorpublish.desktop.vp.i8n.I8n;
import net.vectorpublish.desktop.vp.ui.ImageKey;

@Named

public class UndoAction extends VPAbstractAction implements DocumentSelectionChangeListener, HistoryGrownListener,
		ForwardHistoryStepListener, BackwardHistoryStepListener, Undo {

	/**
	 * The toolbar to hold the undo button.
	 */
	@Inject
	private final EditMenu edit = null;

	@Inject
	private final ToolBar toolbar = null;

	@Inject
	private final I8n i8n = null;

	private DocumentNode doc;

	@Inject
	private final Set<BackwardHistoryStepListener> listeners = null;

	public UndoAction() {
		super(I8nTextKeyframe.UNDO, I8nTextKeyframe.UNDO_DESC, true);
	}

	@Override
	public void actionPerformed(ActionEvent e) {
		doc.getLastExecutedHistoryStep().rollbackToPrevious();
		for (final BackwardHistoryStepListener listener : listeners) {
			listener.notifyHSUndone();
		}
		// TODO: remove next line when https://jira.spring.io/browse/SPR-8450
		// fixed.
		if (!listeners.contains(this)) {
			notifyHSUndone();
		}
	}

	@Override
	public void notifyDocumentSelected(DocumentNode selectedDocument) {
		doc = selectedDocument;
		updateState();
	}

	@Override
	public void notifyHistoryGrown(HistoryStep<?> incomming, boolean byFileRead) {
		if (!byFileRead) {
			updateState();
		}
	}

	@Override
	public void notifyHSDone() {
		updateState();
	}

	@Override
	public void notifyHSUndone() {
		updateState();
	}

	@PostConstruct
	private final void setup() {
		setIcons(RedoAction.NS, ImageKey.get("undo"));
		edit.add(this);
		toolbar.add(this);
	}

	private void updateState() {
		if (doc == null) {
			setEnabled(false);
		} else if (doc.getLastExecutedHistoryStep() == null) {
			setEnabled(false);
		} else {
			setEnabled(doc.getLastExecutedHistoryStep().canRollback());
		}
	}
}
