/*
 * Copyright (c) 2016, Peter Rader. All rights reserved.
 *  ___ ___               __                 ______         __     __  __         __
 * |   |   |.-----..----.|  |_ .-----..----.|   __ \.--.--.|  |--.|  ||__|.-----.|  |--.
 * |   |   ||  -__||  __||   _||  _  ||   _||    __/|  |  ||  _  ||  ||  ||__ --||     |
 *  \_____/ |_____||____||____||_____||__|  |___|   |_____||_____||__||__||_____||__|__|
 *
 * http://www.gnu.org/licenses/gpl-3.0.html
 */
package net.vectorpublish.desktop.vp;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Locale.LanguageRange;
import java.util.Properties;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import javax.annotation.PreDestroy;
import javax.inject.Inject;
import javax.inject.Named;

import org.apache.commons.io.IOUtils;

import net.vectorpublish.desktop.vp.api.conf.Config;
import net.vectorpublish.desktop.vp.api.ui.Dialog;
import net.vectorpublish.desktop.vp.i8n.I8n;
import net.vectorpublish.desktop.vp.i8n.I8nText;
import net.vectorpublish.desktop.vp.i8n.LanguageChangeListener;
import net.vectorpublish.desktop.vp.log.Log;
import net.vectorpublish.desktop.vp.ui.Namespace;

@Named
public class Translation extends Properties implements I8n {

	public static final Namespace DIALOG_NAMESPACE = Namespace.getNamespace("net.vectorpublish", "system.i8n");

	private static final File localFile = new File("translation.properties");

	@Inject
	private final Log log = null;

	@Inject
	private final Dialog dialog = null;

	@Inject
	private final Config config = null;

	@Inject
	private final LanguageChangeListener[] listeners = null;

	private LanguageRange currentLanguage = LanguageController.FALLBACK_RANGE;

	@Inject
	private final DefaultI8nImageFactory translationServerFactory = null;

	public Translation() {
		super();
		if (localFile.exists()) {
			try (FileReader fr = new FileReader(localFile)) {
				load(fr);
			} catch (final IOException e) {
				e.printStackTrace();
			}
		}
	}

	@Override
	public String getTranslation(I8nText key) {
		if (isEmpty()) {
			String read = config.read(DIALOG_NAMESPACE, "language");
			if (read == null) {
				read = "en-en";
			}
			setLanguage(new LanguageRange(read));
		}
		Namespace namespace = key.getNamespace();
		final String finalKey = namespace + "/" + key.getKey();
		String translation = getProperty(finalKey);
		if (translation == null) {
			URL url;
			try {
				url = new URL(translationServerFactory.getTranslationServer() + "rest/properties/" + currentLanguage.getRange() + "/" + namespace + "/"
						+ key.getKey());
				translation = IOUtils.toString(url);
				put(finalKey, translation);
			} catch (final MalformedURLException e) {
				e.printStackTrace();
			} catch (final IOException e) {
				e.printStackTrace();
			}
		}
		if (translation == null) {
			log.missing(finalKey, "in the translation, got to ask the user.");
			final Future<String> ask = dialog.ask(DIALOG_NAMESPACE, "What do you think is the Translation of '" + key.getKey() + "'?",
					new String[] { key.getKey() });
			final Future<Boolean> store = dialog.ask(DIALOG_NAMESPACE, "Store?", true);
			try {
				put(finalKey, translation = ask.get());
				if (store.get()) {
					final Properties p = new Properties();
					if (localFile.exists()) {
						try (FileReader fr = new FileReader(localFile)) {
							p.load(fr);
							p.setProperty(finalKey, translation);
						} catch (final IOException e) {
							e.printStackTrace();
						}
					}
					p.setProperty(finalKey, translation);
					try (FileWriter fw = new FileWriter(localFile)) {
						p.store(fw, "");
					} catch (final IOException e) {
						e.printStackTrace();
					}
				}
			} catch (InterruptedException | ExecutionException e) {
				e.printStackTrace();
			}
			if (translation == null) {
				return key.getKey();
			}
		}
		return translation;
	}

	@Override
	public void setLanguage(LanguageRange newLanguage) {
		final boolean changed = !newLanguage.getRange().equals(currentLanguage.getRange());
		if (changed) {
			clear();
			currentLanguage = newLanguage;
			for (final LanguageChangeListener listener : listeners) {
				listener.changedTo(newLanguage);
			}
		}
	}

	@PreDestroy
	public void tearDown() throws FileNotFoundException, IOException {
		FileOutputStream out = new FileOutputStream(localFile);
		store(out, "");
		out.close();
	}
}
