package xapi.bytecode;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import xapi.source.api.AccessFlag;

public class MethodInfo {
    ConstPool constPool;
    int accessFlags;
    int name;
    String cachedName;
    int descriptor;
    ArrayList<AttributeInfo> attribute; // may be null

    /**
     * If this value is true, Javassist maintains a <code>StackMap</code> attribute
     * generated by the <code>preverify</code> tool of J2ME (CLDC).  The initial
     * value of this field is <code>false</code>.
     */
    public static boolean doPreverify = false;

    /**
     * The name of constructors: <code>&lt;init&gt</code>.
     */
    public static final String nameInit = "<init>";

    /**
     * The name of class initializer (static initializer):
     * <code>&lt;clinit&gt</code>.
     */
    public static final String nameClinit = "<clinit>";

    private MethodInfo(ConstPool cp) {
        constPool = cp;
        attribute = null;
    }

    /**
     * Constructs a <code>method_info</code> structure. The initial value of
     * <code>access_flags</code> is zero.
     *
     * @param cp
     *            a constant pool table
     * @param methodname
     *            method name
     * @param desc
     *            method descriptor
     * @see Descriptor
     */
    public MethodInfo(ConstPool cp, String methodname, String desc) {
        this(cp);
        accessFlags = 0;
        name = cp.addUtf8Info(methodname);
        cachedName = methodname;
        descriptor = constPool.addUtf8Info(desc);
    }

    MethodInfo(ConstPool cp, DataInputStream in) throws IOException {
        this(cp);
        read(in);
    }

    /**
     * Constructs a copy of <code>method_info</code> structure. Class names
     * appearing in the source <code>method_info</code> are renamed according
     * to <code>classnameMap</code>.
     *
     * <p>
     * Note: only <code>Code</code> and <code>Exceptions</code> attributes
     * are copied from the source. The other attributes are ignored.
     *
     * @param cp
     *            a constant pool table
     * @param methodname
     *            a method name
     * @param src
     *            a source <code>method_info</code>
     * @param classnameMap
     *            specifies pairs of replaced and substituted name.
     * @see Descriptor
     */
    public MethodInfo(ConstPool cp, String methodname, MethodInfo src,
            Map<?, ?> classnameMap) {
        this(cp);
        read(src, methodname, classnameMap);
    }

    /**
     * Returns a string representation of the object.
     */
    @Override
    public String toString() {
        return getName() + " " + getDescriptor();
    }

    /**
     * Copies all constant pool items to a given new constant pool
     * and replaces the original items with the new ones.
     * This is used for garbage collecting the items of removed fields
     * and methods.
     *
     * @param cp    the destination
     */
    void compact(ConstPool cp) {
        name = cp.addUtf8Info(getName());
        descriptor = cp.addUtf8Info(getDescriptor());
        attribute = AttributeInfo.copyAll(attribute, cp);
        constPool = cp;
    }

    void prune(ConstPool cp) {
        ArrayList<AttributeInfo> newAttributes = new ArrayList<AttributeInfo>();

        AttributeInfo invisibleAnnotations
            = getAttribute(AnnotationsAttribute.invisibleTag);
        if (invisibleAnnotations != null) {
            invisibleAnnotations = invisibleAnnotations.copy(cp, null);
            newAttributes.add(invisibleAnnotations);
        }

        AttributeInfo visibleAnnotations
            = getAttribute(AnnotationsAttribute.visibleTag);
        if (visibleAnnotations != null) {
            visibleAnnotations = visibleAnnotations.copy(cp, null);
            newAttributes.add(visibleAnnotations);
        }

        AttributeInfo parameterInvisibleAnnotations
            = getAttribute(ParameterAnnotationsAttribute.invisibleTag);
        if (parameterInvisibleAnnotations != null) {
            parameterInvisibleAnnotations = parameterInvisibleAnnotations.copy(cp, null);
            newAttributes.add(parameterInvisibleAnnotations);
        }

        AttributeInfo parameterVisibleAnnotations
            = getAttribute(ParameterAnnotationsAttribute.visibleTag);
        if (parameterVisibleAnnotations != null) {
            parameterVisibleAnnotations = parameterVisibleAnnotations.copy(cp, null);
            newAttributes.add(parameterVisibleAnnotations);
        }

        AnnotationDefaultAttribute defaultAttribute
             = (AnnotationDefaultAttribute) getAttribute(AnnotationDefaultAttribute.tag);
        if (defaultAttribute != null)
            newAttributes.add(defaultAttribute);


        AttributeInfo signature
            = getAttribute(SignatureAttribute.tag);
        if (signature != null) {
            signature = signature.copy(cp, null);
            newAttributes.add(signature);
        }

        attribute = newAttributes;
        name = cp.addUtf8Info(getName());
        descriptor = cp.addUtf8Info(getDescriptor());
        constPool = cp;
    }

    /**
     * Returns a method name.
     */
    public String getName() {
       if (cachedName == null)
           cachedName = constPool.getUtf8Info(name);

       return cachedName;
    }

    /**
     * Sets a method name.
     */
    public void setName(String newName) {
        name = constPool.addUtf8Info(newName);
        cachedName = newName;
    }

    /**
     * Returns true if this is not a constructor or a class initializer (static
     * initializer).
     */
    public boolean isMethod() {
        String n = getName();
        return !n.equals(nameInit) && !n.equals(nameClinit);
    }

    /**
     * Returns a constant pool table used by this method.
     */
    public ConstPool getConstPool() {
        return constPool;
    }

    /**
     * Returns true if this is a constructor.
     */
    public boolean isConstructor() {
        return getName().equals(nameInit);
    }

    /**
     * Returns true if this is a class initializer (static initializer).
     */
    public boolean isStaticInitializer() {
        return getName().equals(nameClinit);
    }

    /**
     * Returns access flags.
     *
     * @see AccessFlag
     */
    public int getAccessFlags() {
        return accessFlags;
    }

    /**
     * Sets access flags.
     *
     * @see AccessFlag
     */
    public void setAccessFlags(int acc) {
        accessFlags = acc;
    }

    /**
     * Returns a method descriptor.
     *
     * @see Descriptor
     */
    public String getDescriptor() {
        return constPool.getUtf8Info(descriptor);
    }

    /**
     * Sets a method descriptor.
     *
     * @see Descriptor
     */
    public void setDescriptor(String desc) {
        if (!desc.equals(getDescriptor()))
            descriptor = constPool.addUtf8Info(desc);
    }

    /**
     * Returns all the attributes.  The returned <code>List</code> object
     * is shared with this object.  If you add a new attribute to the list,
     * the attribute is also added to the method represented by this
     * object.  If you remove an attribute from the list, it is also removed
     * from the method.
     *
     * @return a list of <code>AttributeInfo</code> objects.
     * @see AttributeInfo
     */
    public List<AttributeInfo> getAttributes() {
        if (attribute == null)
            attribute = new ArrayList<AttributeInfo>();

        return attribute;
    }

    /**
     * Returns the attribute with the specified name. If it is not found, this
     * method returns null.
     *
     * @param name attribute name
     * @return an <code>AttributeInfo</code> object or null.
     * @see #getAttributes()
     */
    public AttributeInfo getAttribute(String name) {
        return AttributeInfo.lookup(attribute, name);
    }

    /**
     * Appends an attribute. If there is already an attribute with the same
     * name, the new one substitutes for it.
     *
     * @see #getAttributes()
     */
    public void addAttribute(AttributeInfo info) {
        if (attribute == null)
            attribute = new ArrayList<AttributeInfo>();

        AttributeInfo.remove(attribute, info.getName());
        attribute.add(info);
    }

    private void read(MethodInfo src, String methodname, Map<?, ?> classnames) {
        ConstPool destCp = constPool;
        accessFlags = src.accessFlags;
        name = destCp.addUtf8Info(methodname);
        cachedName = methodname;
        ConstPool srcCp = src.constPool;
        String desc = srcCp.getUtf8Info(src.descriptor);
        String desc2 = Descriptor.rename(desc, classnames);
        descriptor = destCp.addUtf8Info(desc2);

        attribute = new ArrayList<AttributeInfo>();
    }

    private void read(DataInputStream in) throws IOException {
        accessFlags = in.readUnsignedShort();
        name = in.readUnsignedShort();
        descriptor = in.readUnsignedShort();
        int n = in.readUnsignedShort();
        attribute = new ArrayList<AttributeInfo>();
        for (int i = 0; i < n; ++i)
            attribute.add(AttributeInfo.read(constPool, in));
    }

    void write(DataOutputStream out) throws IOException {
        out.writeShort(accessFlags);
        out.writeShort(name);
        out.writeShort(descriptor);

        if (attribute == null)
            out.writeShort(0);
        else {
            out.writeShort(attribute.size());
            AttributeInfo.writeAll(attribute, out);
        }
    }
}
