/*
 * Decompiled with CFR 0.152.
 */
package net.whitbeck.rdbparser;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.charset.Charset;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import net.whitbeck.rdbparser.AuxField;
import net.whitbeck.rdbparser.DoubleBytes;
import net.whitbeck.rdbparser.Entry;
import net.whitbeck.rdbparser.Eof;
import net.whitbeck.rdbparser.IntSet;
import net.whitbeck.rdbparser.KeyValuePair;
import net.whitbeck.rdbparser.ListpackList;
import net.whitbeck.rdbparser.Lzf;
import net.whitbeck.rdbparser.QuickList;
import net.whitbeck.rdbparser.QuickList2;
import net.whitbeck.rdbparser.ResizeDb;
import net.whitbeck.rdbparser.SelectDb;
import net.whitbeck.rdbparser.SortedSetAsListpack;
import net.whitbeck.rdbparser.SortedSetAsZipList;
import net.whitbeck.rdbparser.ValueType;
import net.whitbeck.rdbparser.ZipList;
import net.whitbeck.rdbparser.ZipMap;

public final class RdbParser
implements AutoCloseable {
    private static final Charset ASCII = Charset.forName("ASCII");
    private static final int EOF = 255;
    private static final int SELECTDB = 254;
    private static final int EXPIRETIME = 253;
    private static final int EXPIRETIME_MS = 252;
    private static final int RESIZEDB = 251;
    private static final int AUX = 250;
    private static final int FREQ = 249;
    private static final int IDLE = 248;
    private static final int MODULE_AUX = 247;
    private static final int BUFFER_SIZE = 8192;
    private final ReadableByteChannel ch;
    private final ByteBuffer buf = ByteBuffer.allocateDirect(8192);
    private int version;
    private long bytesBuffered = 0L;
    private boolean isInitialized = false;
    private KeyValuePair nextEntry = null;
    private boolean hasNext = false;

    public RdbParser(ReadableByteChannel ch) {
        this.ch = ch;
    }

    public RdbParser(Path path) throws IOException {
        this.ch = FileChannel.open(path, StandardOpenOption.READ);
    }

    public RdbParser(File file) throws IOException {
        this(file.toPath());
    }

    public RdbParser(InputStream inputStream) throws IOException {
        this(Channels.newChannel(inputStream));
    }

    public RdbParser(String filename) throws IOException {
        this(new File(filename));
    }

    public int getRdbVersion() {
        return this.version;
    }

    private void fillBuffer() throws IOException {
        this.buf.clear();
        long n = this.ch.read(this.buf);
        if (n == -1L) {
            throw new IOException("Attempting to read past channel end-of-stream.");
        }
        this.bytesBuffered += n;
        this.buf.flip();
    }

    private int readByte() throws IOException {
        if (!this.buf.hasRemaining()) {
            this.fillBuffer();
        }
        return this.buf.get() & 0xFF;
    }

    private int readSignedByte() throws IOException {
        if (!this.buf.hasRemaining()) {
            this.fillBuffer();
        }
        return this.buf.get();
    }

    private byte[] readBytes(int numBytes) throws IOException {
        int rem = numBytes;
        int pos = 0;
        byte[] bs = new byte[numBytes];
        while (rem > 0) {
            int avail = this.buf.remaining();
            if (avail >= rem) {
                this.buf.get(bs, pos, rem);
                pos += rem;
                rem = 0;
                continue;
            }
            this.buf.get(bs, pos, avail);
            pos += avail;
            rem -= avail;
            this.fillBuffer();
        }
        return bs;
    }

    private String readMagicNumber() throws IOException {
        return new String(this.readBytes(5), ASCII);
    }

    private int readVersion() throws IOException {
        return Integer.parseInt(new String(this.readBytes(4), ASCII));
    }

    private void init() throws IOException {
        this.fillBuffer();
        if (!this.readMagicNumber().equals("REDIS")) {
            throw new IllegalStateException("Not a valid redis RDB file");
        }
        this.version = this.readVersion();
        if (this.version < 1 || this.version > 10) {
            throw new IllegalStateException("Unknown version");
        }
        this.nextEntry = new KeyValuePair();
        this.hasNext = true;
        this.isInitialized = true;
    }

    public long bytesParsed() {
        return this.bytesBuffered - (long)this.buf.remaining();
    }

    public Entry readNext() throws IOException {
        int valueType;
        block11: while (true) {
            if (!this.hasNext) {
                if (!this.isInitialized) {
                    this.init();
                    continue;
                }
                return null;
            }
            valueType = this.readByte();
            switch (valueType) {
                case 255: {
                    return this.readEof();
                }
                case 254: {
                    return this.readSelectDb();
                }
                case 251: {
                    return this.readResizeDb();
                }
                case 250: {
                    return this.readAuxField();
                }
                case 253: {
                    this.readExpireTime();
                    continue block11;
                }
                case 252: {
                    this.readExpireTimeMillis();
                    continue block11;
                }
                case 249: {
                    this.readFreq();
                    continue block11;
                }
                case 248: {
                    this.readIdle();
                    continue block11;
                }
                case 247: {
                    throw new UnsupportedOperationException("Redis modules are not supported");
                }
            }
            break;
        }
        this.readEntry(valueType);
        KeyValuePair entry = this.nextEntry;
        this.nextEntry = new KeyValuePair();
        return entry;
    }

    private byte[] readChecksum() throws IOException {
        return this.readBytes(8);
    }

    private byte[] getEmptyChecksum() {
        return new byte[8];
    }

    private Eof readEof() throws IOException {
        byte[] checksum = this.version >= 5 ? this.readChecksum() : this.getEmptyChecksum();
        this.hasNext = false;
        return new Eof(checksum);
    }

    private SelectDb readSelectDb() throws IOException {
        return new SelectDb(this.readLength());
    }

    private ResizeDb readResizeDb() throws IOException {
        return new ResizeDb(this.readLength(), this.readLength());
    }

    private AuxField readAuxField() throws IOException {
        return new AuxField(this.readStringEncoded(), this.readStringEncoded());
    }

    private void readFreq() throws IOException {
        this.nextEntry.freq = this.readByte();
    }

    private void readIdle() throws IOException {
        this.nextEntry.idle = this.readLength();
    }

    private long readLength() throws IOException {
        int firstByte = this.readByte();
        int flag = (firstByte & 0xC0) >> 6;
        if (flag == 0) {
            return firstByte & 0x3F;
        }
        if (flag == 1) {
            return ((long)firstByte & 0x3FL) << 8 | (long)this.readByte() & 0xFFL;
        }
        if (firstByte == 128) {
            byte[] bs = this.readBytes(4);
            return ((long)bs[0] & 0xFFL) << 24 | ((long)bs[1] & 0xFFL) << 16 | ((long)bs[2] & 0xFFL) << 8 | ((long)bs[3] & 0xFFL) << 0;
        }
        if (firstByte == 129) {
            byte[] bs = this.readBytes(8);
            return ((long)bs[0] & 0xFFL) << 56 | ((long)bs[1] & 0xFFL) << 48 | ((long)bs[2] & 0xFFL) << 40 | ((long)bs[3] & 0xFFL) << 32 | ((long)bs[4] & 0xFFL) << 24 | ((long)bs[5] & 0xFFL) << 16 | ((long)bs[6] & 0xFFL) << 8 | ((long)bs[7] & 0xFFL) << 0;
        }
        throw new IllegalStateException("Expected a length, but got a special string encoding.");
    }

    private byte[] readStringEncoded() throws IOException {
        int firstByte = this.readByte();
        int flag = (firstByte & 0xC0) >> 6;
        switch (flag) {
            case 0: {
                int len = firstByte & 0x3F;
                return this.readBytes(len);
            }
            case 1: {
                int len = (firstByte & 0x3F) << 8 | this.readByte() & 0xFF;
                return this.readBytes(len);
            }
            case 2: {
                byte[] bs = this.readBytes(4);
                int len = (bs[0] & 0xFF) << 24 | (bs[1] & 0xFF) << 16 | (bs[2] & 0xFF) << 8 | (bs[3] & 0xFF) << 0;
                if (len < 0) {
                    throw new IllegalStateException("Strings longer than 2147483647bytes are not supported.");
                }
                return this.readBytes(len);
            }
            case 3: {
                return this.readSpecialStringEncoded(firstByte & 0x3F);
            }
        }
        return null;
    }

    private byte[] readInteger8Bits() throws IOException {
        return String.valueOf(this.readSignedByte()).getBytes(ASCII);
    }

    private byte[] readInteger16Bits() throws IOException {
        long val = ((long)this.readByte() & 0xFFL) << 0 | (long)this.readSignedByte() << 8;
        return String.valueOf(val).getBytes(ASCII);
    }

    private byte[] readInteger32Bits() throws IOException {
        byte[] bs = this.readBytes(4);
        long val = (long)bs[3] << 24 | ((long)bs[2] & 0xFFL) << 16 | ((long)bs[1] & 0xFFL) << 8 | ((long)bs[0] & 0xFFL) << 0;
        return String.valueOf(val).getBytes(ASCII);
    }

    private byte[] readLzfString() throws IOException {
        int clen = (int)this.readLength();
        int ulen = (int)this.readLength();
        byte[] src = this.readBytes(clen);
        byte[] dest = new byte[ulen];
        Lzf.expand(src, dest);
        return dest;
    }

    private byte[] readDoubleString() throws IOException {
        int len = this.readByte();
        switch (len) {
            case 255: {
                return DoubleBytes.NEGATIVE_INFINITY;
            }
            case 254: {
                return DoubleBytes.POSITIVE_INFINITY;
            }
            case 253: {
                return DoubleBytes.NaN;
            }
        }
        return this.readBytes(len);
    }

    private byte[] readSpecialStringEncoded(int type) throws IOException {
        switch (type) {
            case 0: {
                return this.readInteger8Bits();
            }
            case 1: {
                return this.readInteger16Bits();
            }
            case 2: {
                return this.readInteger32Bits();
            }
            case 3: {
                return this.readLzfString();
            }
        }
        throw new IllegalStateException("Unknown special encoding: " + type);
    }

    private void readExpireTime() throws IOException {
        this.nextEntry.expireTime = this.readBytes(4);
    }

    private void readExpireTimeMillis() throws IOException {
        this.nextEntry.expireTime = this.readBytes(8);
    }

    private void readEntry(int valueType) throws IOException {
        this.nextEntry.key = this.readStringEncoded();
        switch (valueType) {
            case 0: {
                this.readValue();
                break;
            }
            case 1: {
                this.readList();
                break;
            }
            case 2: {
                this.readSet();
                break;
            }
            case 3: {
                this.readSortedSet();
                break;
            }
            case 4: {
                this.readHash();
                break;
            }
            case 5: {
                this.readSortedSet2();
                break;
            }
            case 6: 
            case 7: {
                throw new UnsupportedOperationException("Redis modules are not supported");
            }
            case 9: {
                this.readZipMap();
                break;
            }
            case 10: {
                this.readZipList();
                break;
            }
            case 11: {
                this.readIntSet();
                break;
            }
            case 12: {
                this.readSortedSetAsZipList();
                break;
            }
            case 13: {
                this.readHashmapAsZipList();
                break;
            }
            case 14: {
                this.readQuickList();
                break;
            }
            case 15: 
            case 19: {
                throw new UnsupportedOperationException("Redis streams are not supported");
            }
            case 16: {
                this.readHashListPack();
                break;
            }
            case 17: {
                this.readZSetListPack();
                break;
            }
            case 18: {
                this.readQuickList2();
                break;
            }
            default: {
                throw new UnsupportedOperationException("Unknown value type: " + valueType);
            }
        }
    }

    private void readZSetListPack() throws IOException {
        this.nextEntry.valueType = ValueType.SORTED_SET_AS_LISTPACK;
        this.nextEntry.values = new SortedSetAsListpack(this.readStringEncoded());
    }

    private void readValue() throws IOException {
        this.nextEntry.valueType = ValueType.VALUE;
        this.nextEntry.values = Arrays.asList(new byte[][]{this.readStringEncoded()});
    }

    private void readList() throws IOException {
        long len = this.readLength();
        if (len > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Lists with more than 2147483647 elements are not supported.");
        }
        int size = (int)len;
        ArrayList<byte[]> list = new ArrayList<byte[]>(size);
        for (int i = 0; i < size; ++i) {
            list.add(this.readStringEncoded());
        }
        this.nextEntry.valueType = ValueType.LIST;
        this.nextEntry.values = list;
    }

    private void readSet() throws IOException {
        long len = this.readLength();
        if (len > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Sets with more than 2147483647 elements are not supported.");
        }
        int size = (int)len;
        ArrayList<byte[]> set = new ArrayList<byte[]>(size);
        for (int i = 0; i < size; ++i) {
            set.add(this.readStringEncoded());
        }
        this.nextEntry.valueType = ValueType.SET;
        this.nextEntry.values = set;
    }

    private void readSortedSet() throws IOException {
        long len = this.readLength();
        if (len > 0x3FFFFFFFL) {
            throw new IllegalArgumentException("SortedSets with more than 1073741823 elements are not supported.");
        }
        int size = (int)len;
        ArrayList<byte[]> valueScoresPairs = new ArrayList<byte[]>(2 * size);
        for (int i = 0; i < size; ++i) {
            valueScoresPairs.add(this.readStringEncoded());
            valueScoresPairs.add(this.readDoubleString());
        }
        this.nextEntry.valueType = ValueType.SORTED_SET;
        this.nextEntry.values = valueScoresPairs;
    }

    private void readSortedSet2() throws IOException {
        long len = this.readLength();
        if (len > 0x3FFFFFFFL) {
            throw new IllegalArgumentException("SortedSets with more than 1073741823 elements are not supported.");
        }
        int size = (int)len;
        ArrayList<byte[]> valueScoresPairs = new ArrayList<byte[]>(2 * size);
        for (int i = 0; i < size; ++i) {
            valueScoresPairs.add(this.readStringEncoded());
            valueScoresPairs.add(this.readBytes(8));
        }
        this.nextEntry.valueType = ValueType.SORTED_SET2;
        this.nextEntry.values = valueScoresPairs;
    }

    private void readHash() throws IOException {
        long len = this.readLength();
        if (len > 0x3FFFFFFFL) {
            throw new IllegalArgumentException("Hashes with more than 1073741823 elements are not supported.");
        }
        int size = (int)len;
        ArrayList<byte[]> kvPairs = new ArrayList<byte[]>(2 * size);
        for (int i = 0; i < size; ++i) {
            kvPairs.add(this.readStringEncoded());
            kvPairs.add(this.readStringEncoded());
        }
        this.nextEntry.valueType = ValueType.HASH;
        this.nextEntry.values = kvPairs;
    }

    private void readZipMap() throws IOException {
        this.nextEntry.valueType = ValueType.ZIPMAP;
        this.nextEntry.values = new ZipMap(this.readStringEncoded());
    }

    private void readZipList() throws IOException {
        this.nextEntry.valueType = ValueType.ZIPLIST;
        this.nextEntry.values = new ZipList(this.readStringEncoded());
    }

    private void readIntSet() throws IOException {
        this.nextEntry.valueType = ValueType.INTSET;
        this.nextEntry.values = new IntSet(this.readStringEncoded());
    }

    private void readSortedSetAsZipList() throws IOException {
        this.nextEntry.valueType = ValueType.SORTED_SET_AS_ZIPLIST;
        this.nextEntry.values = new SortedSetAsZipList(this.readStringEncoded());
    }

    private void readHashmapAsZipList() throws IOException {
        this.nextEntry.valueType = ValueType.HASHMAP_AS_ZIPLIST;
        this.nextEntry.values = new ZipList(this.readStringEncoded());
    }

    private void readHashListPack() throws IOException {
        this.nextEntry.valueType = ValueType.HASHMAP_AS_LISTPACK;
        this.nextEntry.values = new ListpackList(this.readStringEncoded());
    }

    private void readQuickList2() throws IOException {
        int size = (int)this.readLength();
        ArrayList<byte[]> listpacks = new ArrayList<byte[]>(size);
        for (int i = 0; i < size; ++i) {
            this.readLength();
            listpacks.add(this.readStringEncoded());
        }
        this.nextEntry.valueType = ValueType.QUICKLIST2;
        this.nextEntry.values = new QuickList2(listpacks);
    }

    private void readQuickList() throws IOException {
        long len = this.readLength();
        if (len > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Quicklists with more than 2147483647 nested Ziplists are not supported.");
        }
        int size = (int)len;
        ArrayList<byte[]> ziplists = new ArrayList<byte[]>(size);
        for (int i = 0; i < size; ++i) {
            ziplists.add(this.readStringEncoded());
        }
        this.nextEntry.valueType = ValueType.QUICKLIST;
        this.nextEntry.values = new QuickList(ziplists);
    }

    @Override
    public void close() throws IOException {
        this.ch.close();
    }

    public static double parseSortedSetScore(byte[] bytes) {
        return Double.parseDouble(new String(bytes, ASCII));
    }

    public static double parseSortedSet2Score(byte[] bytes) {
        return ByteBuffer.wrap(bytes).order(ByteOrder.LITTLE_ENDIAN).getDouble();
    }
}

