package net.wombyte.buffer;

import net.wombyte.WombyteHttpClient;
import net.wombyte.event.WombyteEventContext;
import net.wombyte.tracker.WombyteTelemetry;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

public class WombyteBuffer implements Closeable {

    private final ReadWriteLock writeLock = new ReentrantReadWriteLock();
    private final ReadWriteLock sendLock = new ReentrantReadWriteLock();

    private final File directory;
    private final WombyteHttpClient client;
    private final WombyteTelemetry telemetry;

    private WombyteBufferWriter writer;

    public WombyteBuffer(File directory, WombyteHttpClient client, WombyteTelemetry telemetry) {
        this.directory = directory;
        this.client = client;
        this.telemetry = telemetry;

        this.writer = createWriter();
    }

    private WombyteBufferWriter createWriter() {
        UUID uuid = UUID.randomUUID();

        return new WombyteBufferWriter(directory.toString() + "/" + uuid);
    }

    private File[] rotate() {
        File[] files = directory.listFiles();

        writeLock.writeLock().lock();

        try {
            writer.close();
            writer = createWriter();
        } finally {
            writeLock.writeLock().unlock();
        }

        return files;
    }

    public CompletableFuture<Boolean> append(WombyteEventContext context, Map<String, Object> attributes) {
        Object[] record = new Object[]
                {
                        context.timestamp,
                        context.id,
                        context.event,
                        context.app,

                        attributes
                };

        return CompletableFuture.supplyAsync(() -> {
            writeLock.writeLock().lock();

            try {
                writer.write(record);
            } finally {
                writeLock.writeLock().unlock();
            }

            return true;
        });

    }

    @Override
    public void close() {
        writer.close();
    }

    public void flushLines() {
        sendLock.writeLock().lock();

        try {
            for (File file : rotate()) {
                Instant start = Instant.now();
                int response = client.sendFile(file);
                if (response < 500) {
                    BasicFileAttributes attr = Files.readAttributes(file.toPath(), BasicFileAttributes.class);
                    telemetry.logBatchTimeRange(attr.creationTime().toInstant(), attr.lastModifiedTime().toInstant());
                    telemetry.logBatchSize(Files.size(file.toPath()));
                    telemetry.logServerResponse(Instant.now(), response, Duration.between(start, Instant.now()).toMillis());
                    Files.delete(file.toPath());
                } else {
                    telemetry.logServerResponse(Instant.now(), response, Duration.between(start, Instant.now()).toMillis());
                }
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            sendLock.writeLock().unlock();
        }
    }


}
