package multiplatform.network.cmptoast

import android.app.Activity
import android.content.Context
import android.graphics.drawable.GradientDrawable
import android.os.Build
import android.util.Log
import android.view.Gravity
import android.view.LayoutInflater
import android.widget.TextView
import android.widget.Toast
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.toArgb
import androidx.compose.ui.unit.TextUnit
import androidx.compose.ui.unit.sp


fun toast(
    context: Context,
    message: String = "",
    duration: ToastDuration = ToastDuration.Short,
    gravity: ToastGravity = ToastGravity.Bottom,
    backgroundColor: Color? = Color.Black,
    textColor: Color? = Color.White,
    padding: ToastPadding? = null,
    cornerRadius: Int? = 100,
    textSize: TextUnit = 14.sp,
    topPadding: Int = 100,
    bottomPadding: Int = 100
) {
    val mToast: Toast?

    val mGravity: Int = when (gravity) {
        ToastGravity.Top -> {
            Gravity.TOP
        }
        ToastGravity.Center -> {
            Gravity.CENTER
        }
        ToastGravity.Bottom -> {
            Gravity.BOTTOM
        }
    }

    val toastDuration = when(duration) {
        ToastDuration.Long -> Toast.LENGTH_LONG
        ToastDuration.Short -> Toast.LENGTH_SHORT
    }

    if (backgroundColor != null) {
        val layout = (context.getSystemService(Context.LAYOUT_INFLATER_SERVICE) as LayoutInflater).inflate(R.layout.toast_custom, null)
        val text = layout.findViewById<TextView>(R.id.text)
        text.text = message

        // Create a new shape drawable with the custom corner radius
        val shapeDrawable = GradientDrawable()
        shapeDrawable.shape = GradientDrawable.RECTANGLE
        shapeDrawable.setColor(backgroundColor.toArgb())
        shapeDrawable.cornerRadius = cornerRadius?.toFloat() ?: 100f

        // Set toast inside padding
        val toastPadding = padding ?: ToastPadding(top = 30, bottom = 30, left = 60, right = 60)
        layout.background = shapeDrawable

        layout.setPadding(toastPadding.left, toastPadding.top, toastPadding.right, toastPadding.bottom)
        text.textSize = textSize.value

        if (textColor != null) {
            text.setTextColor(textColor.toArgb())
        }

        mToast = Toast(context)
        mToast.duration = toastDuration

        mToast.view = layout
    } else {
        mToast = Toast.makeText(context, message, toastDuration)
        mToast.duration = toastDuration
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.R) {
            val textView: TextView = mToast?.view!!.findViewById(android.R.id.message)
            textView.textSize = textSize.value
            if (textColor != null) {
                textView.setTextColor(textColor.toArgb())
            }
        }
    }

    try {
        when (mGravity) {
            Gravity.CENTER -> {
                mToast?.setGravity(mGravity, 0, 0)
            }
            Gravity.TOP -> {
                mToast?.setGravity(mGravity, 0, topPadding)
            }
            else -> {
                mToast?.setGravity(mGravity, 0, bottomPadding)
            }
        }
    } catch (e: Exception,) {
        Log.e("Toast", "Exception : ${e.message}")
    }

    if (context is Activity) {
        context.runOnUiThread { mToast?.show() }
    } else {
        mToast?.show()
    }
}