package multiplatform.network.cmptoast

import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.toArgb
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.TextUnit
import androidx.compose.ui.unit.sp
import kotlinx.cinterop.ExperimentalForeignApi
import kotlinx.cinterop.memScoped
import kotlinx.cinterop.pointed
import platform.UIKit.NSTextAlignmentCenter
import platform.UIKit.NSTextAlignmentJustified
import platform.UIKit.NSTextAlignmentLeft
import platform.UIKit.NSTextAlignmentRight
import platform.UIKit.UIApplication
import platform.UIKit.UIColor
import platform.UIKit.UIFont
import platform.UIKit.UILabel
import platform.UIKit.UIView
import platform.UIKit.UIWindow

fun toast(
    message: String = "",
    duration: ToastDuration = ToastDuration.Short,
    cornerRadius: Int? = 15,
    gravity: ToastGravity = ToastGravity.Bottom,
    textAlignment: TextAlign = TextAlign.Center,
    textColor: Color = Color.White,
    backgroundColor: Color = Color.Black,
    padding: ToastPadding? = null,
    textSize: TextUnit = 12.sp,
    topPadding: Int = 10,
    bottomPadding: Int = 10
) {
    val containerView = UIView()
    containerView.backgroundColor = getUIColor(backgroundColor)
    containerView.layer.cornerRadius = cornerRadius?.toDouble() ?: 18.0
    containerView.alpha = 1.0
    containerView.layer.masksToBounds = true
    containerView.clipsToBounds = true

    val toastView = UILabel()
    toastView.textColor = getUIColor(textColor)
    toastView.text = message
    toastView.textAlignment = when(textAlignment) {
        TextAlign.Left, TextAlign.Start -> NSTextAlignmentLeft
        TextAlign.Right, TextAlign.End -> NSTextAlignmentRight
        TextAlign.Center -> NSTextAlignmentCenter
        TextAlign.Justify -> NSTextAlignmentJustified
        else -> NSTextAlignmentLeft
    }
    toastView.font = UIFont.systemFontOfSize(textSize.value.toDouble())
    toastView.layer.masksToBounds = true
    toastView.clipsToBounds = true
    toastView.numberOfLines = 0

    // Set toast inside padding
    val toastPadding = padding ?: ToastPadding(top = 10, bottom = 10, left = 10, right = 10)

    val window = UIApplication.sharedApplication.keyWindow()
    window?.addSubview(containerView)
    containerView.translatesAutoresizingMaskIntoConstraints = false
    window?.leadingAnchor?.let { containerView.leadingAnchor.constraintGreaterThanOrEqualToAnchor(it, constant = 30.0).active = true }
    window?.trailingAnchor?.let { containerView.trailingAnchor.constraintLessThanOrEqualToAnchor(it, constant = -(30.0)).active = true }
    window?.centerXAnchor?.let { containerView.centerXAnchor.constraintEqualToAnchor(it).active = true }

    when(gravity) {
        ToastGravity.Bottom -> {
            window?.bottomAnchor?.let { containerView.bottomAnchor.constraintEqualToAnchor(it, constant = -(bottomPadding.toDouble() + getSafeAreaHeight().bottom)).active = true }
        }
        ToastGravity.Center -> {
            window?.centerYAnchor?.let { containerView.centerYAnchor.constraintEqualToAnchor(it).active = true }
        }
        ToastGravity.Top -> {
            window?.topAnchor?.let { containerView.topAnchor.constraintEqualToAnchor(it, constant = (topPadding.toDouble() + getSafeAreaHeight().top)).active = true }
        }
    }

    containerView.addSubview(toastView)
    toastView.translatesAutoresizingMaskIntoConstraints = false
    containerView.leadingAnchor.let { toastView.leadingAnchor.constraintEqualToAnchor(it, constant = toastPadding.left.toDouble()).active = true }
    containerView.trailingAnchor.let { toastView.trailingAnchor.constraintEqualToAnchor(it, constant = -(toastPadding.right.toDouble())).active = true }
    containerView.bottomAnchor.let { toastView.bottomAnchor.constraintEqualToAnchor(it, constant = -(toastPadding.bottom.toDouble())).active = true }
    containerView.topAnchor.let { toastView.topAnchor.constraintEqualToAnchor(it, constant = toastPadding.top.toDouble()).active = true }

    val toastDuration = when(duration) {
        ToastDuration.Long -> 5.0
        ToastDuration.Short -> 3.0
    }

    UIView.animateWithDuration(toastDuration, animations = { containerView.alpha = 0.0 }, completion = {
        containerView.removeFromSuperview()
    })
}

fun getUIColor(color: Color): UIColor {
    val colorInt = color.toArgb()
    val a = (colorInt shr 24) and 0xff
    val r = (colorInt shr 16) and 0xff
    val g = (colorInt shr 8) and 0xff
    val b = colorInt and 0xff

    return UIColor(
        red = r / 255.0,
        green = g / 255.0,
        blue = b / 255.0,
        alpha = a / 255.0
    )
}

@OptIn(ExperimentalForeignApi::class)
fun getSafeAreaHeight(): SafeAreaSize {
    val window = UIApplication.sharedApplication.windows.first() as UIWindow
    val safeAreaInsets = window.safeAreaInsets
    val topPadding = memScoped {
        val top = safeAreaInsets.getPointer(this)
        top.pointed.top
    }
    val bottomPadding = memScoped {
        val bottom = safeAreaInsets.getPointer(this)
        bottom.pointed.bottom
    }

    return SafeAreaSize(top = topPadding.toFloat(), bottom = bottomPadding.toFloat())
}

data class SafeAreaSize(
    var top: Float = 0f,
    var bottom: Float = 0f,
)