package multiplatform.network.cmptoast

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.fadeIn
import androidx.compose.animation.fadeOut
import androidx.compose.animation.slideInVertically
import androidx.compose.animation.slideOutVertically
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Popup
import androidx.compose.ui.window.PopupProperties
import kotlinx.coroutines.delay

@Composable
fun ToastHost() {
    val currentToast by ToastManager.currentToast

    currentToast?.let { toast ->
        ToastComponent(
            toast = toast,
            onDismiss = { ToastManager.dismissToast() }
        )
    }
}

@Composable
private fun ToastComponent(
    toast: ToastData,
    onDismiss: () -> Unit
) {
    val density = LocalDensity.current
    var visible by remember { mutableStateOf(false) }

    LaunchedEffect(toast) {
        visible = true
        delay(
            when (toast.duration) {
                ToastDuration.Short -> 2000L
                ToastDuration.Long -> 4000L
            }
        )
        visible = false
        delay(300) // Animation duration
        onDismiss()
    }

    val alignment = when (toast.gravity) {
        ToastGravity.Top -> Alignment.TopCenter
        ToastGravity.Center -> Alignment.Center
        ToastGravity.Bottom -> Alignment.BottomCenter
    }

    val verticalOffset = with(density) {
        when (toast.gravity) {
            ToastGravity.Top -> toast.topPadding.dp
            ToastGravity.Bottom -> (-toast.bottomPadding).dp
            ToastGravity.Center -> 0.dp
        }
    }

    Popup(
        alignment = alignment,
        offset = androidx.compose.ui.unit.IntOffset(0, verticalOffset.toPx(density)),
        properties = PopupProperties(
            dismissOnBackPress = false,
            dismissOnClickOutside = false,
            clippingEnabled = false
        )
    ) {
        AnimatedVisibility(
            visible = visible,
            enter = slideInVertically { height ->
                when (toast.gravity) {
                    ToastGravity.Top -> -height
                    ToastGravity.Bottom -> height
                    ToastGravity.Center -> 0
                }
            } + fadeIn(),
            exit = slideOutVertically { height ->
                when (toast.gravity) {
                    ToastGravity.Top -> -height
                    ToastGravity.Bottom -> height
                    ToastGravity.Center -> 0
                }
            } + fadeOut()
        ) {
            Box(
                modifier = Modifier
                    .padding(horizontal = 16.dp)
                    .clip(
                        RoundedCornerShape(
                            toast.cornerRadius?.dp ?: 30.dp
                        )
                    )
                    .background(toast.backgroundColor)
                    .padding(
                        top = toast.padding?.top?.dp ?: 16.dp,
                        bottom = toast.padding?.bottom?.dp ?: 12.dp,
                        start = toast.padding?.left?.dp ?: 12.dp,
                        end = toast.padding?.right?.dp ?: 12.dp,
                    )
            ) {
                Row(
                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.Center
                ) {
                    toast.icon?.let { icon ->
                        Image(
                            bitmap = icon,
                            contentDescription = null,
                            modifier = Modifier
                                .size(toast.iconSizeDp.dp)
                                .padding(end = 8.dp)
                        )
                    }

                    Text(
                        text = toast.message,
                        color = toast.textColor,
                        fontSize = toast.textSize,
                        fontWeight = FontWeight.Medium
                    )
                }
            }
        }
    }
}