@file:OptIn(ExperimentalForeignApi::class)

package multiplatform.network.cmptoast

import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ImageBitmap
import androidx.compose.ui.graphics.toArgb
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.TextUnit
import androidx.compose.ui.unit.sp
import kotlinx.cinterop.ExperimentalForeignApi
import kotlinx.cinterop.memScoped
import kotlinx.cinterop.pointed
import platform.UIKit.NSTextAlignmentCenter
import platform.UIKit.NSTextAlignmentJustified
import platform.UIKit.NSTextAlignmentLeft
import platform.UIKit.NSTextAlignmentRight
import platform.UIKit.UIApplication
import platform.UIKit.UIColor
import platform.UIKit.UIFont
import platform.UIKit.UIImageView
import platform.UIKit.UILabel
import platform.UIKit.UILayoutConstraintAxisHorizontal
import platform.UIKit.UIStackView
import platform.UIKit.UIStackViewAlignmentCenter
import platform.UIKit.UIView
import platform.UIKit.UIViewContentMode
import platform.UIKit.UIWindow

fun toast(
    message: String = "",
    duration: ToastDuration = ToastDuration.Short,
    cornerRadius: Int? = 15,
    gravity: ToastGravity = ToastGravity.Bottom,
    textAlignment: TextAlign = TextAlign.Center,
    textColor: Color = Color.White,
    backgroundColor: Color = Color.Black,
    padding: ToastPadding? = null,
    textSize: TextUnit = 12.sp,
    topPadding: Int = 10,
    bottomPadding: Int = 10,
    icon: ImageBitmap? = null,
    iconSizeDp: Int = 20
) {
    val containerView = UIView()
    containerView.backgroundColor = getUIColor(backgroundColor)
    containerView.layer.cornerRadius = cornerRadius?.toDouble() ?: 18.0
    containerView.alpha = 1.0
    containerView.layer.masksToBounds = true
    containerView.clipsToBounds = true

    val stackView = UIStackView()
    stackView.axis = UILayoutConstraintAxisHorizontal
    stackView.alignment = UIStackViewAlignmentCenter
    stackView.spacing = 8.0
    stackView.translatesAutoresizingMaskIntoConstraints = false

    // Optional icon view
    icon?.let { iconBitmap ->
        try {
            // Use the extension function to convert ImageBitmap to UIImage
            val uiImage = iconBitmap.toUIImage()
            val imageView = UIImageView(image = uiImage)
            imageView.contentMode = UIViewContentMode.UIViewContentModeScaleAspectFit
            imageView.translatesAutoresizingMaskIntoConstraints = false

            // Use the iconSizeDp parameter for consistent sizing
            val iconSize = iconSizeDp.toDouble()
            imageView.widthAnchor.constraintEqualToConstant(iconSize).active = true
            imageView.heightAnchor.constraintEqualToConstant(iconSize).active = true
            stackView.addArrangedSubview(imageView)
        } catch (e: Exception) {
            // Log error or handle gracefully - toast will show without icon
            println("Failed to add icon to toast: ${e.message}")
        }
    }

    val toastLabel = UILabel()
    toastLabel.textColor = getUIColor(textColor)
    toastLabel.text = message
    toastLabel.textAlignment = when (textAlignment) {
        TextAlign.Left, TextAlign.Start -> NSTextAlignmentLeft
        TextAlign.Right, TextAlign.End -> NSTextAlignmentRight
        TextAlign.Center -> NSTextAlignmentCenter
        TextAlign.Justify -> NSTextAlignmentJustified
        else -> NSTextAlignmentLeft
    }
    toastLabel.font = UIFont.systemFontOfSize(textSize.value.toDouble())
    toastLabel.numberOfLines = 0
    toastLabel.clipsToBounds = true
    stackView.addArrangedSubview(toastLabel)

    val toastPadding = padding ?: ToastPadding(top = 10, bottom = 10, left = 10, right = 10)

    containerView.addSubview(stackView)

    val window = UIApplication.sharedApplication.keyWindow()
    window?.addSubview(containerView)

    containerView.translatesAutoresizingMaskIntoConstraints = false
    stackView.translatesAutoresizingMaskIntoConstraints = false

    // Constraints for containerView
    window?.leadingAnchor?.let {
        containerView.leadingAnchor.constraintGreaterThanOrEqualToAnchor(it, constant = 30.0).active = true
    }
    window?.trailingAnchor?.let {
        containerView.trailingAnchor.constraintLessThanOrEqualToAnchor(it, constant = -30.0).active = true
    }
    window?.centerXAnchor?.let {
        containerView.centerXAnchor.constraintEqualToAnchor(it).active = true
    }

    when (gravity) {
        ToastGravity.Bottom -> {
            window?.bottomAnchor?.let {
                containerView.bottomAnchor.constraintEqualToAnchor(
                    it,
                    constant = -(bottomPadding.toDouble() + getSafeAreaHeight().bottom)
                ).active = true
            }
        }
        ToastGravity.Center -> {
            window?.centerYAnchor?.let {
                containerView.centerYAnchor.constraintEqualToAnchor(it).active = true
            }
        }
        ToastGravity.Top -> {
            window?.topAnchor?.let {
                containerView.topAnchor.constraintEqualToAnchor(
                    it,
                    constant = (topPadding.toDouble() + getSafeAreaHeight().top)
                ).active = true
            }
        }
    }

    // Constraints for stackView inside container
    stackView.topAnchor.constraintEqualToAnchor(
        containerView.topAnchor,
        constant = toastPadding.top.toDouble()
    ).active = true
    stackView.bottomAnchor.constraintEqualToAnchor(
        containerView.bottomAnchor,
        constant = -toastPadding.bottom.toDouble()
    ).active = true
    stackView.leadingAnchor.constraintEqualToAnchor(
        containerView.leadingAnchor,
        constant = toastPadding.left.toDouble()
    ).active = true
    stackView.trailingAnchor.constraintEqualToAnchor(
        containerView.trailingAnchor,
        constant = -toastPadding.right.toDouble()
    ).active = true

    val toastDuration = when (duration) {
        ToastDuration.Long -> 5.0
        ToastDuration.Short -> 3.0
    }

    UIView.animateWithDuration(toastDuration, animations = {
        containerView.alpha = 0.0
    }, completion = { _ ->
        containerView.removeFromSuperview()
    })
}

fun getUIColor(color: Color): UIColor {
    val colorInt = color.toArgb()
    val a = (colorInt shr 24) and 0xff
    val r = (colorInt shr 16) and 0xff
    val g = (colorInt shr 8) and 0xff
    val b = colorInt and 0xff

    return UIColor(
        red = r / 255.0,
        green = g / 255.0,
        blue = b / 255.0,
        alpha = a / 255.0
    )
}

@OptIn(ExperimentalForeignApi::class)
fun getSafeAreaHeight(): SafeAreaSize {
    val window = UIApplication.sharedApplication.windows.first() as UIWindow
    val safeAreaInsets = window.safeAreaInsets
    val topPadding = memScoped {
        val top = safeAreaInsets.getPointer(this)
        top.pointed.top
    }
    val bottomPadding = memScoped {
        val bottom = safeAreaInsets.getPointer(this)
        bottom.pointed.bottom
    }

    return SafeAreaSize(top = topPadding.toFloat(), bottom = bottomPadding.toFloat())
}

data class SafeAreaSize(
    var top: Float = 0f,
    var bottom: Float = 0f,
)