package network.chaintech.kmp_date_time_picker.ui.datetimepicker

import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.height
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import kotlinx.datetime.LocalDate
import network.chaintech.kmp_date_time_picker.utils.MAX
import network.chaintech.kmp_date_time_picker.utils.MIN
import network.chaintech.kmp_date_time_picker.utils.Month
import network.chaintech.kmp_date_time_picker.utils.Year
import network.chaintech.kmp_date_time_picker.utils.calculateDayOfMonths
import network.chaintech.kmp_date_time_picker.utils.now
import network.chaintech.kmp_date_time_picker.utils.shortMonths
import network.chaintech.kmp_date_time_picker.utils.withDayOfMonth
import network.chaintech.kmp_date_time_picker.utils.withMonth
import network.chaintech.kmp_date_time_picker.utils.withYear
import network.chaintech.ui.datetimepicker.SnappedDate

@Composable
internal fun DefaultWheelDatePicker(
    modifier: Modifier = Modifier,
    startDate: LocalDate = LocalDate.now(),
    minDate: LocalDate = LocalDate.MIN(),
    maxDate: LocalDate = LocalDate.MAX(),
    yearsRange: IntRange? = IntRange(1922, 2122),
    height: Dp = 128.dp,
    rowCount: Int = 3,
    textStyle: TextStyle = MaterialTheme.typography.titleMedium,
    textColor: Color = LocalContentColor.current,
    showMonthAsNumber: Boolean = false, // Added flag to show month as number or string
    onSnappedDate: (snappedDate: SnappedDate) -> Int? = { _ -> null }
) {
    var snappedDate by remember { mutableStateOf(startDate) }

    var dayOfMonths = calculateDayOfMonths(
        snappedDate.monthNumber,
        snappedDate.year
    )

    // Modify months list to display either as numbers or as short month names
    val months = (1..12).map {
        Month(
            text = if (showMonthAsNumber) it.toString() else shortMonths(it), // Display logic
            value = it,
            index = it - 1
        )
    }

    val years = yearsRange?.map {
        Year(
            text = it.toString(),
            value = it,
            index = yearsRange.indexOf(it)
        )
    }

    Box(modifier = modifier, contentAlignment = Alignment.Center) {
        Row {
            years?.let { years ->
                WheelTextPicker(
                    modifier = Modifier.weight(1f),
                    startIndex = years.find { it.value == startDate.year }?.index ?: 0,
                    height = height,
                    texts = years.map { it.text },
                    rowCount = rowCount,
                    style = textStyle,
                    color = textColor
                ) { snappedIndex ->

                    val newYear = years.find { it.index == snappedIndex }?.value

                    newYear?.let {

                        val newDate = snappedDate.withYear(newYear)

                        if (newDate.compareTo(minDate) >= 0 && newDate.compareTo(maxDate) <= 0) {
                            snappedDate = newDate
                        }

                        dayOfMonths =
                            calculateDayOfMonths(
                                snappedDate.monthNumber,
                                snappedDate.year
                            )

                        val newIndex = years.find { it.value == snappedDate.year }?.index

                        newIndex?.let {
                            onSnappedDate(
                                SnappedDate.Year(
                                    localDate = snappedDate,
                                    index = newIndex
                                )
                            )?.let { return@WheelTextPicker it }

                        }
                    }

                    return@WheelTextPicker years.find { it.value == snappedDate.year }?.index
                }
            }
            Box(
                modifier = Modifier
                    .height(height),
                contentAlignment = Alignment.Center
            ) {
                Text(
                    text = "-",
                    style = textStyle,
                    color = textColor
                )
            }
            WheelTextPicker(
                modifier = Modifier.weight(1f),
                startIndex = months.find { it.value == startDate.monthNumber }?.index ?: 0,
                height = height,
                texts = months.map { it.text },
                rowCount = rowCount,
                style = textStyle,
                color = textColor
            ) { snappedIndex ->

                val newMonth = months.find { it.index == snappedIndex }?.value

                newMonth?.let {

                    val newDate = snappedDate.withMonth(newMonth)

                    if (newDate.compareTo(minDate) >= 0 && newDate.compareTo(maxDate) <= 0) {
                        snappedDate = newDate
                    }

                    dayOfMonths =
                        calculateDayOfMonths(
                            snappedDate.monthNumber,
                            snappedDate.year
                        )

                    val newIndex = months.find { it.value == snappedDate.monthNumber }?.index

                    newIndex?.let {
                        onSnappedDate(
                            SnappedDate.Month(
                                localDate = snappedDate,
                                index = newIndex
                            )
                        )?.let { return@WheelTextPicker it }
                    }
                }


                return@WheelTextPicker months.find { it.value == snappedDate.monthNumber }?.index
            }
            Box(
                modifier = Modifier
                    .height(height),
                contentAlignment = Alignment.Center
            ) {
                Text(
                    text = "-",
                    style = textStyle,
                    color = textColor
                )
            }
            WheelTextPicker(
                modifier = Modifier.weight(1f),
                startIndex = dayOfMonths.find { it.value == startDate.dayOfMonth }?.index ?: 0,
                height = height,
                texts = dayOfMonths.map { it.text },
                rowCount = rowCount,
                style = textStyle,
                color = textColor
            ) { snappedIndex ->

                val newDayOfMonth = dayOfMonths.find { it.index == snappedIndex }?.value

                newDayOfMonth?.let {
                    val newDate = snappedDate.withDayOfMonth(newDayOfMonth)

                    if (newDate.compareTo(minDate) >= 0 && newDate.compareTo(maxDate) <= 0) {
                        snappedDate = newDate
                    }

                    val newIndex =
                        dayOfMonths.find { it.value == snappedDate.dayOfMonth }?.index

                    newIndex?.let {
                        onSnappedDate(
                            SnappedDate.DayOfMonth(
                                localDate = snappedDate,
                                index = newIndex
                            )
                        )?.let { return@WheelTextPicker it }
                    }
                }

                return@WheelTextPicker dayOfMonths.find { it.value == snappedDate.dayOfMonth }?.index
            }
        }
    }
}
