package network.chaintech.kmp_date_time_picker.ui.timepicker

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import kotlinx.datetime.DateTimeUnit
import kotlinx.datetime.LocalTime
import network.chaintech.kmp_date_time_picker.utils.AmPm
import network.chaintech.kmp_date_time_picker.utils.AmPmHour
import network.chaintech.kmp_date_time_picker.utils.AmPmValue
import network.chaintech.kmp_date_time_picker.utils.Hour
import network.chaintech.kmp_date_time_picker.utils.MAX
import network.chaintech.kmp_date_time_picker.utils.MIN
import network.chaintech.kmp_date_time_picker.utils.Minute
import network.chaintech.kmp_date_time_picker.utils.SelectorProperties
import network.chaintech.kmp_date_time_picker.utils.TimeFormat
import network.chaintech.kmp_date_time_picker.utils.WheelPickerDefaults
import network.chaintech.kmp_date_time_picker.utils.amPmHourToHour24
import network.chaintech.kmp_date_time_picker.utils.amPmValueFromTime
import network.chaintech.kmp_date_time_picker.utils.localTimeToAmPmHour
import network.chaintech.kmp_date_time_picker.utils.now
import network.chaintech.kmp_date_time_picker.utils.truncateTo
import network.chaintech.kmp_date_time_picker.utils.withHour
import network.chaintech.kmp_date_time_picker.utils.withMinute

@Composable
internal fun DefaultWheelTimePicker(
    modifier: Modifier = Modifier,
    startTime: LocalTime = LocalTime.now(),
    minTime: LocalTime = LocalTime.MIN(),
    maxTime: LocalTime = LocalTime.MAX(),
    timeFormat: TimeFormat = TimeFormat.HOUR_24,
    height: Dp,
    rowCount: Int = 3,
    textStyle: TextStyle = MaterialTheme.typography.titleMedium,
    textColor: Color = LocalContentColor.current,
    selectorProperties: SelectorProperties = WheelPickerDefaults.selectorProperties(borderColor = Color.Transparent),
    onSnappedTime: (snappedTime: SnappedTime, timeFormat: TimeFormat) -> Int? = { _, _ -> null },
) {

    var snappedTime by remember { mutableStateOf(startTime.truncateTo(DateTimeUnit.MINUTE)) }

    val hours = (0..23).map {
        Hour(
            text = it.toString().padStart(2, '0'),
            value = it,
            index = it
        )
    }
    val amPmHours = (1..12).map {
        AmPmHour(
            text = it.toString(),
            value = it,
            index = it - 1
        )
    }

    val minutes = (0..59).map {
        Minute(
            text = it.toString().padStart(2, '0'),
            value = it,
            index = it
        )
    }

    val amPms = listOf(
        AmPm(
            text = "AM",
            value = AmPmValue.AM,
            index = 0
        ),
        AmPm(
            text = "PM",
            value = AmPmValue.PM,
            index = 1
        )
    )

    var snappedAmPm by remember {
        mutableStateOf(
            amPms.find { it.value == amPmValueFromTime(startTime) } ?: amPms[0]
        )
    }

    Box(
        modifier = modifier.fillMaxWidth(),
        contentAlignment = Alignment.Center
    ) {
        if (selectorProperties.enabled().value) {
            HorizontalDivider(
                modifier = Modifier.padding(bottom = (height / rowCount)),
                thickness = (0.5).dp,
                color = selectorProperties.borderColor().value
            )
            HorizontalDivider(
                modifier = Modifier.padding(top = (height / rowCount)),
                thickness = (0.5).dp,
                color = selectorProperties.borderColor().value
            )
        }
        Row(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.Center
        ) {
            WheelTextPicker(
                modifier = Modifier.width(80.dp),
                startIndex = if (timeFormat == TimeFormat.HOUR_24) {
                    hours.find { it.value == startTime.hour }?.index ?: 0
                } else amPmHours.find { it.value == localTimeToAmPmHour(startTime) }?.index
                    ?: 0,
                height = height,
                texts = if (timeFormat == TimeFormat.HOUR_24) hours.map { it.text } else amPmHours.map { it.text },
                rowCount = rowCount,
                style = textStyle,
                color = textColor
            ) { snappedIndex ->

                val newHour = if (timeFormat == TimeFormat.HOUR_24) {
                    hours.find { it.index == snappedIndex }?.value
                } else {
                    amPmHourToHour24(
                        amPmHours.find { it.index == snappedIndex }?.value ?: 0,
                        snappedTime.minute,
                        snappedAmPm.value
                    )
                }

                newHour?.let {

                    val newTime = snappedTime.withHour(newHour)

                    if (newTime.compareTo(minTime) >= 0 && newTime.compareTo(maxTime) <= 0) {
                        snappedTime = newTime
                    }

                    val newIndex = if (timeFormat == TimeFormat.HOUR_24) {
                        hours.find { it.value == snappedTime.hour }?.index
                    } else {
                        amPmHours.find { it.value == localTimeToAmPmHour(snappedTime) }?.index
                    }

                    newIndex?.let {
                        onSnappedTime(
                            SnappedTime.Hour(
                                localTime = snappedTime,
                                index = newIndex
                            ),
                            timeFormat
                        )?.let { return@WheelTextPicker it }
                    }
                }

                return@WheelTextPicker if (timeFormat == TimeFormat.HOUR_24) {
                    hours.find { it.value == snappedTime.hour }?.index
                } else {
                    amPmHours.find { it.value == localTimeToAmPmHour(snappedTime) }?.index
                }
            }
            Box(
                modifier = Modifier
                    .height(height),
                contentAlignment = Alignment.Center
            ) {
                Text(
                    text = ":",
                    style = textStyle,
                    color = textColor
                )
            }
            WheelTextPicker(
                modifier = Modifier.width(80.dp),
                startIndex = minutes.find { it.value == startTime.minute }?.index ?: 0,
                height = height,
                texts = minutes.map { it.text },
                rowCount = rowCount,
                style = textStyle,
                color = textColor
            ) { snappedIndex ->

                val newMinute = minutes.find { it.index == snappedIndex }?.value

                val newHour = if (timeFormat == TimeFormat.HOUR_24) {
                    hours.find { it.value == snappedTime.hour }?.value
                } else {
                    amPmHourToHour24(
                        amPmHours.find { it.value == localTimeToAmPmHour(snappedTime) }?.value
                            ?: 0,
                        snappedTime.minute,
                        snappedAmPm.value
                    )
                }

                newMinute?.let {
                    newHour?.let {
                        val newTime = snappedTime.withMinute(newMinute).withHour(newHour)

                        if (newTime.compareTo(minTime) >= 0 && newTime.compareTo(maxTime) <= 0) {
                            snappedTime = newTime
                        }

                        val newIndex =
                            minutes.find { it.value == snappedTime.minute }?.index

                        newIndex?.let {
                            onSnappedTime(
                                SnappedTime.Minute(
                                    localTime = snappedTime,
                                    index = newIndex
                                ),
                                timeFormat
                            )?.let { return@WheelTextPicker it }
                        }
                    }
                }

                return@WheelTextPicker minutes.find { it.value == snappedTime.minute }?.index
            }
            if (timeFormat == TimeFormat.AM_PM) {
                WheelTextPicker(
                    modifier = Modifier.width(50.dp),
                    startIndex = amPms.find { it.value == amPmValueFromTime(startTime) }?.index
                        ?: 0,
                    height = height,
                    texts = amPms.map { it.text },
                    rowCount = rowCount,
                    style = textStyle,
                    color = textColor
                ) { snappedIndex ->

                    val newAmPm = amPms.find {
                        if (snappedIndex == 2) {
                            it.index == 1
                        } else {
                            it.index == snappedIndex
                        }
                    }

                    newAmPm?.let {
                        snappedAmPm = newAmPm
                    }

                    val newMinute = minutes.find { it.value == snappedTime.minute }?.value

                    val newHour = amPmHourToHour24(
                        amPmHours.find { it.value == localTimeToAmPmHour(snappedTime) }?.value
                            ?: 0,
                        snappedTime.minute,
                        snappedAmPm.value
                    )

                    newMinute?.let {
                        val newTime = snappedTime.withMinute(newMinute).withHour(newHour)

                        if (newTime.compareTo(minTime) >= 0 && newTime.compareTo(maxTime) <= 0) {
                            snappedTime = newTime
                        }

                        val newIndex = minutes.find { it.value == snappedTime.hour }?.index

                        newIndex?.let {
                            onSnappedTime(
                                SnappedTime.Hour(
                                    localTime = snappedTime,
                                    index = newIndex
                                ),
                                timeFormat
                            )
                        }
                    }

                    return@WheelTextPicker snappedIndex
                }
            }
        }
    }
}