package network.chaintech.kmp_date_time_picker.ui.timepicker

import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.BottomSheetDefaults
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.LocalTextStyle
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.Shape
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import kotlinx.datetime.LocalTime
import network.chaintech.kmp_date_time_picker.utils.DateTimePickerView
import network.chaintech.kmp_date_time_picker.utils.MAX
import network.chaintech.kmp_date_time_picker.utils.MIN
import network.chaintech.kmp_date_time_picker.utils.SelectorProperties
import network.chaintech.kmp_date_time_picker.utils.TimeFormat
import network.chaintech.kmp_date_time_picker.utils.WheelPickerDefaults
import network.chaintech.kmp_date_time_picker.utils.now

/***
 * modifier: Modifies the layout of the time picker.
 * showTimePicker: Show and hide time picker.
 * title: Title displayed above the time picker.
 * doneLabel: Label for the "Done" button.
 * titleStyle: Style for the title text.
 * doneLabelStyle: Style for the "Done" label text.
 * startTime: Initial time selected in the picker.
 * minTime: Minimum selectable time.
 * maxTime: Maximum selectable time.
 * timeFormat: Format for displaying time (e.g., 24-hour format).
 * height: height of the time picker component.
 * rowCount: Number of rows displayed in the picker and it's depending on height also.
 * textStyle: Text style for the time display.
 * textColor: Text color for the time display.
 * hideHeader: Hide header of picker.
 * containerColor: The color used for the background of time picker.
 * shape: The shape of the time picker.
 * dateTimePickerView: For bottomsheet and diloag view.
 * dragHandle - Optional visual marker to swipe the bottom sheet.
 * selectorProperties: Properties defining the interaction with the time picker.
 * onDoneClick: Callback triggered when the "Done" button is clicked, passing the selected time.
 * onTimeChangeListener: Callback triggered when the time is changed, passing the selected time.
 * onDismiss: Callback triggered when the time picker is dismissed.
 ***/

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun WheelTimePickerView(
    modifier: Modifier = Modifier,
    showTimePicker: Boolean = false,
    title: String = "TIME PICKER",
    doneLabel: String = "Done",
    titleStyle: TextStyle = LocalTextStyle.current,
    doneLabelStyle: TextStyle = LocalTextStyle.current,
    startTime: LocalTime = LocalTime.now(),
    minTime: LocalTime = LocalTime.MIN(),
    maxTime: LocalTime = LocalTime.MAX(),
    timeFormat: TimeFormat = TimeFormat.HOUR_24,
    height: Dp,
    rowCount: Int = 3,
    textStyle: TextStyle = MaterialTheme.typography.titleMedium,
    textColor: Color = LocalContentColor.current,
    hideHeader: Boolean = false,
    containerColor: Color = Color.White,
    shape: Shape = RoundedCornerShape(10.dp),
    dateTimePickerView: DateTimePickerView = DateTimePickerView.BOTTOM_SHEET_VIEW,
    dragHandle: @Composable (() -> Unit)? = { BottomSheetDefaults.DragHandle() },
    selectorProperties: SelectorProperties = WheelPickerDefaults.selectorProperties(),
    onDoneClick: (snappedDate: LocalTime) -> Unit = {},
    onTimeChangeListener: (snappedDate: LocalTime) -> Unit = {},
    onDismiss: () -> Unit = {},
) {
    if (dateTimePickerView == DateTimePickerView.BOTTOM_SHEET_VIEW) {
        WheelTimePickerBottomSheet(
            modifier = modifier,
            showTimePicker = showTimePicker,
            title = title,
            doneLabel = doneLabel,
            titleStyle = titleStyle,
            doneLabelStyle = doneLabelStyle,
            textColor = textColor,
            timeFormat = timeFormat,
            selectorProperties = selectorProperties,
            rowCount = rowCount,
            height = height,
            hideHeader = hideHeader,
            onDoneClick = onDoneClick,
            startTime = startTime,
            minTime = minTime,
            maxTime = maxTime,
            textStyle = textStyle,
            onDismiss = onDismiss,
            containerColor = containerColor,
            shape = shape,
            dragHandle = dragHandle,
            onTimeChangeListener = onTimeChangeListener,
        )
    } else {
        WheelTimePickerDialog(
            modifier = modifier,
            showTimePicker = showTimePicker,
            title = title,
            doneLabel = doneLabel,
            titleStyle = titleStyle,
            doneLabelStyle = doneLabelStyle,
            textColor = textColor,
            timeFormat = timeFormat,
            selectorProperties = selectorProperties,
            rowCount = rowCount,
            height = height,
            hideHeader = hideHeader,
            onDoneClick = onDoneClick,
            startTime = startTime,
            minTime = minTime,
            maxTime = maxTime,
            textStyle = textStyle,
            onDismiss = onDismiss,
            containerColor = containerColor,
            shape = shape,
            onTimeChangeListener = onTimeChangeListener,
        )
    }
}

