/*
   Copyright 2021 Barend Garvelink

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
 */
package nl.garvelink.iban;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;

/**
 * Contains information about IBAN country codes. This is a generated file.
 * Updated to SWIFT IBAN Registry version 97 on 2024-05-25.
 */
abstract class CountryCodesData {
    /**
     * The "yyyy-MM-dd" datestamp that the embedded IBAN data was updated.
     */
    static final String LAST_UPDATE_DATE = "2024-05-25";
    /**
     * The revision of the SWIFT IBAN Registry to which the embedded IBAN data was updated.
     */
    static final String LAST_UPDATE_REV = "97";

    static final int SEPA = 1 << 8;
    static final int SWIFT = 1 << 9;
    static final int REMOVE_METADATA_MASK = 0xFF;
    static final int BANK_IDENTIFIER_BEGIN_MASK = 0xFF;
    static final int BANK_IDENTIFIER_END_SHIFT = 8;
    static final int BANK_IDENTIFIER_END_MASK = 0xFF << BANK_IDENTIFIER_END_SHIFT;
    static final int BRANCH_IDENTIFIER_BEGIN_SHIFT = 16;
    static final int BRANCH_IDENTIFIER_BEGIN_MASK = 0xFF << BRANCH_IDENTIFIER_BEGIN_SHIFT;
    static final int BRANCH_IDENTIFIER_END_SHIFT = 24;
    static final int BRANCH_IDENTIFIER_END_MASK = 0xFF << BRANCH_IDENTIFIER_END_SHIFT;

    /**
     * Known country codes, this list must be sorted to allow binary search. All other lists in this file must use the
     * same indices for the same countries.
     */
    static final String[] COUNTRY_CODES = {
        "AD",
        "AE",
        "AL",
        "AO",
        "AT",
        "AZ",
        "BA",
        "BE",
        "BF",
        "BG",
        "BH",
        "BI",
        "BJ",
        "BR",
        "BY",
        "CF",
        "CG",
        "CH",
        "CI",
        "CM",
        "CR",
        "CV",
        "CY",
        "CZ",
        "DE",
        "DJ",
        "DK",
        "DO",
        "DZ",
        "EE",
        "EG",
        "ES",
        "FI",
        "FK",
        "FO",
        "FR",
        "GA",
        "GB",
        "GE",
        "GI",
        "GL",
        "GQ",
        "GR",
        "GT",
        "GW",
        "HN",
        "HR",
        "HU",
        "IE",
        "IL",
        "IQ",
        "IR",
        "IS",
        "IT",
        "JO",
        "KM",
        "KW",
        "KZ",
        "LB",
        "LC",
        "LI",
        "LT",
        "LU",
        "LV",
        "LY",
        "MA",
        "MC",
        "MD",
        "ME",
        "MG",
        "MK",
        "ML",
        "MN",
        "MR",
        "MT",
        "MU",
        "MZ",
        "NE",
        "NI",
        "NL",
        "NO",
        "OM",
        "PK",
        "PL",
        "PS",
        "PT",
        "QA",
        "RO",
        "RS",
        "RU",
        "SA",
        "SC",
        "SD",
        "SE",
        "SI",
        "SK",
        "SM",
        "SN",
        "SO",
        "ST",
        "SV",
        "TD",
        "TG",
        "TL",
        "TN",
        "TR",
        "UA",
        "VA",
        "VG",
        "XK"
    };

    /**
     * Lengths for each country's IBAN. The indices match the indices of {@link #COUNTRY_CODES}, the values are the
     * expected length. Values may embed the {@link #SEPA} and {@link #SWIFT} flags to indicate the SEPA membership and
     * whether the record is listed in the SWIFT IBAN Registry.
     */
    static final int[] COUNTRY_IBAN_LENGTHS = {
        /* AD */ 24 | SWIFT | SEPA,
        /* AE */ 23 | SWIFT,
        /* AL */ 28 | SWIFT,
        /* AO */ 25,
        /* AT */ 20 | SWIFT | SEPA,
        /* AZ */ 28 | SWIFT,
        /* BA */ 20 | SWIFT,
        /* BE */ 16 | SWIFT | SEPA,
        /* BF */ 28,
        /* BG */ 22 | SWIFT | SEPA,
        /* BH */ 22 | SWIFT,
        /* BI */ 27 | SWIFT,
        /* BJ */ 28,
        /* BR */ 29 | SWIFT,
        /* BY */ 28 | SWIFT,
        /* CF */ 27,
        /* CG */ 27,
        /* CH */ 21 | SWIFT | SEPA,
        /* CI */ 28,
        /* CM */ 27,
        /* CR */ 22 | SWIFT,
        /* CV */ 25,
        /* CY */ 28 | SWIFT | SEPA,
        /* CZ */ 24 | SWIFT | SEPA,
        /* DE */ 22 | SWIFT | SEPA,
        /* DJ */ 27 | SWIFT,
        /* DK */ 18 | SWIFT | SEPA,
        /* DO */ 28 | SWIFT,
        /* DZ */ 26,
        /* EE */ 20 | SWIFT | SEPA,
        /* EG */ 29 | SWIFT,
        /* ES */ 24 | SWIFT | SEPA,
        /* FI */ 18 | SWIFT | SEPA,
        /* FK */ 18 | SWIFT,
        /* FO */ 18 | SWIFT,
        /* FR */ 27 | SWIFT | SEPA,
        /* GA */ 27,
        /* GB */ 22 | SWIFT | SEPA,
        /* GE */ 22 | SWIFT,
        /* GI */ 23 | SWIFT | SEPA,
        /* GL */ 18 | SWIFT,
        /* GQ */ 27,
        /* GR */ 27 | SWIFT | SEPA,
        /* GT */ 28 | SWIFT,
        /* GW */ 25,
        /* HN */ 28,
        /* HR */ 21 | SWIFT | SEPA,
        /* HU */ 28 | SWIFT | SEPA,
        /* IE */ 22 | SWIFT | SEPA,
        /* IL */ 23 | SWIFT,
        /* IQ */ 23 | SWIFT,
        /* IR */ 26,
        /* IS */ 26 | SWIFT | SEPA,
        /* IT */ 27 | SWIFT | SEPA,
        /* JO */ 30 | SWIFT,
        /* KM */ 27,
        /* KW */ 30 | SWIFT,
        /* KZ */ 20 | SWIFT,
        /* LB */ 28 | SWIFT,
        /* LC */ 32 | SWIFT,
        /* LI */ 21 | SWIFT | SEPA,
        /* LT */ 20 | SWIFT | SEPA,
        /* LU */ 20 | SWIFT | SEPA,
        /* LV */ 21 | SWIFT | SEPA,
        /* LY */ 25 | SWIFT,
        /* MA */ 28,
        /* MC */ 27 | SWIFT | SEPA,
        /* MD */ 24 | SWIFT,
        /* ME */ 22 | SWIFT,
        /* MG */ 27,
        /* MK */ 19 | SWIFT,
        /* ML */ 28,
        /* MN */ 20 | SWIFT,
        /* MR */ 27 | SWIFT,
        /* MT */ 31 | SWIFT | SEPA,
        /* MU */ 30 | SWIFT,
        /* MZ */ 25,
        /* NE */ 28,
        /* NI */ 28 | SWIFT,
        /* NL */ 18 | SWIFT | SEPA,
        /* NO */ 15 | SWIFT | SEPA,
        /* OM */ 23 | SWIFT,
        /* PK */ 24 | SWIFT,
        /* PL */ 28 | SWIFT | SEPA,
        /* PS */ 29 | SWIFT,
        /* PT */ 25 | SWIFT | SEPA,
        /* QA */ 29 | SWIFT,
        /* RO */ 24 | SWIFT | SEPA,
        /* RS */ 22 | SWIFT,
        /* RU */ 33 | SWIFT,
        /* SA */ 24 | SWIFT,
        /* SC */ 31 | SWIFT,
        /* SD */ 18 | SWIFT,
        /* SE */ 24 | SWIFT | SEPA,
        /* SI */ 19 | SWIFT | SEPA,
        /* SK */ 24 | SWIFT | SEPA,
        /* SM */ 27 | SWIFT | SEPA,
        /* SN */ 28,
        /* SO */ 23 | SWIFT,
        /* ST */ 25 | SWIFT,
        /* SV */ 28 | SWIFT,
        /* TD */ 27,
        /* TG */ 28,
        /* TL */ 23 | SWIFT,
        /* TN */ 24 | SWIFT,
        /* TR */ 26 | SWIFT,
        /* UA */ 29 | SWIFT,
        /* VA */ 22 | SWIFT | SEPA,
        /* VG */ 24 | SWIFT,
        /* XK */ 20 | SWIFT
    };


    /**
     * Contains the start- and end-index (as per {@link String#substring(int, int)}) of the bank code and branch code
     * within a country's IBAN format. Mask:
     * <pre>
     * 0x000000FF <- begin offset bank id
     * 0x0000FF00 <- end offset bank id
     * 0x00FF0000 <- begin offset branch id
     * 0xFF000000 <- end offset branch id
     * </pre>
     */
    static final int[] BANK_CODE_BRANCH_CODE = {
        /* AD */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 8 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (8 + 4) << BRANCH_IDENTIFIER_END_SHIFT,

        /* AE */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* AL */
        4
        | (4 + 8) << BANK_IDENTIFIER_END_SHIFT
        | 7 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (7 + 4) << BRANCH_IDENTIFIER_END_SHIFT,

        /* AO */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* AT */
        4
        | (4 + 5) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* AZ */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* BA */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 7 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (7 + 3) << BRANCH_IDENTIFIER_END_SHIFT,

        /* BE */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* BF */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* BG */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 8 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (8 + 4) << BRANCH_IDENTIFIER_END_SHIFT,

        /* BH */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* BI */
        4
        | (4 + 5) << BANK_IDENTIFIER_END_SHIFT
        | 9 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (9 + 5) << BRANCH_IDENTIFIER_END_SHIFT,

        /* BJ */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* BR */
        4
        | (4 + 8) << BANK_IDENTIFIER_END_SHIFT
        | 12 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (12 + 5) << BRANCH_IDENTIFIER_END_SHIFT,

        /* BY */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* CF */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* CG */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* CH */
        4
        | (4 + 5) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* CI */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* CM */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* CR */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* CV */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* CY */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 7 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (7 + 5) << BRANCH_IDENTIFIER_END_SHIFT,

        /* CZ */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* DE */
        4
        | (4 + 8) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* DJ */
        4
        | (4 + 5) << BANK_IDENTIFIER_END_SHIFT
        | 9 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (9 + 5) << BRANCH_IDENTIFIER_END_SHIFT,

        /* DK */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* DO */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* DZ */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* EE */
        4
        | (4 + 2) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* EG */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 8 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (8 + 4) << BRANCH_IDENTIFIER_END_SHIFT,

        /* ES */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 8 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (8 + 4) << BRANCH_IDENTIFIER_END_SHIFT,

        /* FI */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* FK */
        4
        | (4 + 2) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* FO */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* FR */
        4
        | (4 + 5) << BANK_IDENTIFIER_END_SHIFT
        | 9 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (9 + 5) << BRANCH_IDENTIFIER_END_SHIFT,

        /* GA */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* GB */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 8 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (8 + 6) << BRANCH_IDENTIFIER_END_SHIFT,

        /* GE */
        4
        | (4 + 2) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* GI */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* GL */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* GQ */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* GR */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 7 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (7 + 4) << BRANCH_IDENTIFIER_END_SHIFT,

        /* GT */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* GW */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* HN */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* HR */
        4
        | (4 + 7) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* HU */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 7 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (7 + 4) << BRANCH_IDENTIFIER_END_SHIFT,

        /* IE */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 8 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (8 + 6) << BRANCH_IDENTIFIER_END_SHIFT,

        /* IL */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 7 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (7 + 3) << BRANCH_IDENTIFIER_END_SHIFT,

        /* IQ */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 8 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (8 + 3) << BRANCH_IDENTIFIER_END_SHIFT,

        /* IR */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* IS */
        4
        | (4 + 2) << BANK_IDENTIFIER_END_SHIFT
        | 6 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (6 + 2) << BRANCH_IDENTIFIER_END_SHIFT,

        /* IT */
        5
        | (5 + 5) << BANK_IDENTIFIER_END_SHIFT
        | 10 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (10 + 5) << BRANCH_IDENTIFIER_END_SHIFT,

        /* JO */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* KM */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* KW */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* KZ */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* LB */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* LC */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* LI */
        4
        | (4 + 5) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* LT */
        4
        | (4 + 5) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* LU */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* LV */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* LY */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 7 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (7 + 3) << BRANCH_IDENTIFIER_END_SHIFT,

        /* MA */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* MC */
        4
        | (4 + 5) << BANK_IDENTIFIER_END_SHIFT
        | 9 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (9 + 5) << BRANCH_IDENTIFIER_END_SHIFT,

        /* MD */
        4
        | (4 + 2) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* ME */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* MG */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* MK */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* ML */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* MN */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* MR */
        4
        | (4 + 5) << BANK_IDENTIFIER_END_SHIFT
        | 9 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (9 + 5) << BRANCH_IDENTIFIER_END_SHIFT,

        /* MT */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 8 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (8 + 5) << BRANCH_IDENTIFIER_END_SHIFT,

        /* MU */
        4
        | (4 + 6) << BANK_IDENTIFIER_END_SHIFT
        | 10 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (10 + 2) << BRANCH_IDENTIFIER_END_SHIFT,

        /* MZ */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* NE */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* NI */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* NL */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* NO */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* OM */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* PK */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* PL */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 4 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (4 + 8) << BRANCH_IDENTIFIER_END_SHIFT,

        /* PS */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* PT */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* QA */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* RO */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* RS */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* RU */
        4
        | (4 + 9) << BANK_IDENTIFIER_END_SHIFT
        | 13 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (13 + 5) << BRANCH_IDENTIFIER_END_SHIFT,

        /* SA */
        4
        | (4 + 2) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* SC */
        4
        | (4 + 6) << BANK_IDENTIFIER_END_SHIFT
        | 10 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (10 + 2) << BRANCH_IDENTIFIER_END_SHIFT,

        /* SD */
        4
        | (4 + 2) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* SE */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* SI */
        4
        | (4 + 5) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* SK */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* SM */
        5
        | (5 + 5) << BANK_IDENTIFIER_END_SHIFT
        | 10 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (10 + 5) << BRANCH_IDENTIFIER_END_SHIFT,

        /* SN */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* SO */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 8 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (8 + 3) << BRANCH_IDENTIFIER_END_SHIFT,

        /* ST */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 8 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (8 + 4) << BRANCH_IDENTIFIER_END_SHIFT,

        /* SV */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* TD */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* TG */
        0
        | (0 + 0) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* TL */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* TN */
        4
        | (4 + 2) << BANK_IDENTIFIER_END_SHIFT
        | 6 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (6 + 3) << BRANCH_IDENTIFIER_END_SHIFT,

        /* TR */
        4
        | (4 + 5) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* UA */
        4
        | (4 + 6) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* VA */
        4
        | (4 + 3) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* VG */
        4
        | (4 + 4) << BANK_IDENTIFIER_END_SHIFT
        | 0 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (0 + 0) << BRANCH_IDENTIFIER_END_SHIFT,

        /* XK */
        4
        | (4 + 2) << BANK_IDENTIFIER_END_SHIFT
        | 6 << BRANCH_IDENTIFIER_BEGIN_SHIFT
        | (6 + 2) << BRANCH_IDENTIFIER_END_SHIFT

    };

    /** Prevent instantiation of static utility class. */
    private CountryCodesData() { }
}
