/* ========================================================================
 * PlantUML : a free UML diagram generator
 * ========================================================================
 *
 * (C) Copyright 2009-2023, Arnaud Roques
 *
 * Project Info:  https://plantuml.com
 * 
 * If you like this project or if you find it useful, you can support us at:
 * 
 * https://plantuml.com/patreon (only 1$ per month!)
 * https://plantuml.com/paypal
 * 
 * This file is part of PlantUML.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 * Original Author:  Arnaud Roques
 */
package net.sourceforge.plantuml.braille;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import net.sourceforge.plantuml.awt.geom.XCubicCurve2D;
import net.sourceforge.plantuml.awt.geom.XPoint2D;
import net.sourceforge.plantuml.posimo.DotPath;

public class BrailleGrid {

	private int minX;
	private int minY;
	private int maxX;
	private int maxY;

	private final double quanta;
	private final Set<Coords> on = new HashSet<>();

	public BrailleGrid(double quanta) {
		this.quanta = quanta;
	}

	public boolean getState(int x, int y) {
		final Coords coords = new Coords(x, y);
		return on.contains(coords);
	}

	private void setStateDouble(double x, double y, boolean state) {
		setState(toInt(x), toInt(y), state);
	}

	public void setState(int x, int y, boolean state) {
		final Coords coords = new Coords(x, y);
		if (state) {
			on.add(coords);
		} else {
			on.remove(coords);
		}
		minX = Math.min(minX, x);
		maxX = Math.max(maxX, x);
		minY = Math.min(minY, y);
		maxY = Math.max(maxY, y);
	}

	public int getMinX() {
		return minX;
	}

	public int getMinY() {
		return minY;
	}

	public int getMaxX() {
		return maxX;
	}

	public int getMaxY() {
		return maxY;
	}

	public void rectangle(double x, double y, double width, double height) {
		hline(y, x, x + width);
		hline(y + height, x, x + width);
		vline(x, y, y + height);
		vline(x + width, y, y + height);

	}

	private void vline(double x, double y1, double y2) {
		final int i = toInt(x);
		final int j1 = toInt(y1);
		final int j2 = toInt(y2);
		for (int j = j1; j <= j2; j++) {
			setState(i, j, true);
		}
	}

	private void hline(double y, double x1, double x2) {
		final int j = toInt(y);
		final int i1 = toInt(x1);
		final int i2 = toInt(x2);
		for (int i = i1; i <= i2; i++) {
			setState(i, j, true);
		}
	}

	public int toInt(double value) {
		return (int) Math.round(value / quanta);
	}

	public void line(double x1, double y1, double x2, double y2) {
		if (x1 == x2) {
			vline(x1, y1, y2);
		} else if (y1 == y2) {
			hline(y1, x1, x2);
		} else {
			System.err.println("warning line");
		}

	}

	public double getQuanta() {
		return quanta;
	}

	public void drawDotPath(double x, double y, DotPath shape) {
		for (XCubicCurve2D bez : shape.getBeziers()) {
			drawCubic(x, y, bez);

		}
	}

	private void drawCubic(double x, double y, XCubicCurve2D bez) {
		drawPointInternal(x, y, bez.getP1());
		drawPointInternal(x, y, bez.getP2());
		if (bez.getP1().distance(bez.getP2()) > quanta) {
			final XCubicCurve2D part1 = new XCubicCurve2D();
			final XCubicCurve2D part2 = new XCubicCurve2D();
			bez.subdivide(part1, part2);
			drawCubic(x, y, part1);
			drawCubic(x, y, part2);
		}
	}

	private void drawPointInternal(double x, double y, XPoint2D pt) {
		setStateDouble(x + pt.getX(), y + pt.getY(), true);
	}

	public void drawPolygon(List<XPoint2D> points) {
		for (int i = 0; i < points.size() - 1; i++) {
			drawLineInternal(points.get(i), points.get(i + 1));
		}
		drawLineInternal(points.get(points.size() - 1), points.get(0));

	}

	private void drawLineInternal(XPoint2D a, XPoint2D b) {
		drawPointInternal(0, 0, a);
		drawPointInternal(0, 0, b);
		if (a.distance(b) > quanta) {
			final XPoint2D middle = new XPoint2D((a.getX() + b.getX()) / 2, (a.getY() + b.getY()) / 2);
			drawLineInternal(a, middle);
			drawLineInternal(middle, b);

		}
	}
}
