/*
 * Copyright 2016-2022 Talsma ICT
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package nl.talsmasoftware.umldoclet.html;

import nl.talsmasoftware.umldoclet.configuration.Configuration;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Collection;
import java.util.List;

import static java.util.Objects.requireNonNull;
import static nl.talsmasoftware.umldoclet.logging.Message.DEBUG_REPLACING_BY;
import static nl.talsmasoftware.umldoclet.logging.Message.DEBUG_SKIPPING_FILE;
import static nl.talsmasoftware.umldoclet.logging.Message.INFO_ADD_DIAGRAM_TO_FILE;

/**
 * Abstraction for a single HTML file generated by the Standard doclet.
 *
 * @author Sjoerd Talsma
 */
final class HtmlFile {

    final Configuration config;
    final Path path;

    HtmlFile(Configuration config, Path path) {
        this.config = requireNonNull(config, "Configuration is <null>.");
        this.path = requireNonNull(path, "HTML file is <null>.").normalize();
    }

    static boolean isHtmlFile(Path path) {
        final File file = path == null ? null : path.toFile();
        return file != null && file.isFile() && file.canRead() && file.getName().endsWith(".html");
    }

    boolean process(Collection<DiagramFile> diagrams) {
        return diagrams.stream()
                .filter(diagram -> diagram.matches(this))
                .map(diagram -> new Postprocessor(this, diagram))
                .map(this::process)
                .reduce(Boolean.FALSE, Boolean::logicalOr)
                || skip();
    }

    private boolean skip() {
        config.logger().debug(DEBUG_SKIPPING_FILE, path);
        return false;
    }

    private boolean process(Postprocessor postprocessor) {
        try {
            if (postprocessor.call()) {
                config.logger().info(INFO_ADD_DIAGRAM_TO_FILE, path);
                return true;
            }
        } catch (IOException ioe) {
            throw new IllegalStateException("I/O exception postprocessing " + path, ioe);
        }
        return false; // should log skip after all diagram types were processed
    }

    public List<String> readLines() throws IOException {
        return Files.readAllLines(path, config.htmlCharset());
    }

    File createNewTempFile() throws IOException {
        final String fileName = path.getFileName().toString();
        final int lastDot = fileName.lastIndexOf('.');
        String prefix = fileName, suffix = null;
        if (lastDot > 0) {
            prefix = fileName.substring(0, lastDot);
            suffix = fileName.substring(lastDot);
        }
        prefix += "---".substring(0, Math.max(1, 3 - prefix.length())); // Make sure the prefix is at least 3 chars.
        return File.createTempFile(prefix, suffix);
    }

    public void replaceBy(File tempFile) throws IOException {
        if (tempFile != null && tempFile.isFile()) {
            Files.move(tempFile.toPath(), path, StandardCopyOption.REPLACE_EXISTING);
            config.logger().debug(DEBUG_REPLACING_BY, path, tempFile);
        }
    }
}
