/*
 * Decompiled with CFR 0.152.
 */
package no.nav.common.auth.oidc.filter;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.proc.BadJOSEException;
import com.nimbusds.jwt.JWT;
import com.nimbusds.jwt.JWTParser;
import com.nimbusds.openid.connect.sdk.validators.BadJWTExceptions;
import java.text.ParseException;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import no.nav.common.auth.oidc.TokenRefreshClient;
import no.nav.common.auth.oidc.filter.OidcAuthenticator;
import no.nav.common.auth.subject.SsoToken;
import no.nav.common.auth.subject.Subject;
import no.nav.common.auth.subject.SubjectHandler;
import no.nav.common.auth.utils.CookieUtils;
import no.nav.common.auth.utils.TokenUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OidcAuthenticationFilter
implements Filter {
    private static final Logger logger = LoggerFactory.getLogger(OidcAuthenticationFilter.class);
    private static final long CHECK_EXPIRES_WITHIN = 300000L;
    private final List<OidcAuthenticator> oidcAuthenticators;
    private final TokenRefreshClient tokenRefreshClient;

    public OidcAuthenticationFilter(List<OidcAuthenticator> oidcAuthenticators) {
        this.oidcAuthenticators = oidcAuthenticators;
        this.tokenRefreshClient = new TokenRefreshClient();
    }

    public void init(FilterConfig filterConfig) {
    }

    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) {
        HttpServletRequest httpServletRequest = (HttpServletRequest)request;
        HttpServletResponse httpServletResponse = (HttpServletResponse)response;
        for (OidcAuthenticator authenticator : this.oidcAuthenticators) {
            Optional<String> token = authenticator.findIdToken(httpServletRequest);
            if (!token.isPresent()) continue;
            try {
                JWT jwtToken = JWTParser.parse((String)token.get());
                Optional<String> refreshedIdToken = this.refreshIdTokenIfNecessary(jwtToken, authenticator, httpServletRequest);
                if (refreshedIdToken.isPresent()) {
                    jwtToken = JWTParser.parse((String)refreshedIdToken.get());
                    String idTokenCookieName = authenticator.config.idTokenCookieName;
                    this.addNewIdTokenCookie(idTokenCookieName, jwtToken, httpServletRequest, httpServletResponse);
                }
                authenticator.tokenValidator.validate(jwtToken);
                SsoToken ssoToken = SsoToken.oidcToken(jwtToken.getParsedString(), jwtToken.getJWTClaimsSet().getClaims());
                Subject subject = new Subject(TokenUtils.getUid(jwtToken, authenticator.config.identType), authenticator.config.identType, ssoToken);
                SubjectHandler.withSubject(subject, () -> chain.doFilter(request, response));
                return;
            }
            catch (JOSEException | BadJOSEException | ParseException exception) {
                if (exception == BadJWTExceptions.EXPIRED_EXCEPTION) {
                    logger.info("Token validation failed", exception);
                    continue;
                }
                logger.error("Token validation failed", exception);
            }
        }
        httpServletResponse.setStatus(401);
    }

    private void addNewIdTokenCookie(String idTokenCookieName, JWT jwtToken, HttpServletRequest request, HttpServletResponse response) throws ParseException {
        Date cookieExpiration = jwtToken.getJWTClaimsSet().getExpirationTime();
        Cookie newIdCookie = CookieUtils.createCookie(idTokenCookieName, jwtToken.getParsedString(), cookieExpiration, request);
        response.addCookie(newIdCookie);
    }

    private Optional<String> refreshIdTokenIfNecessary(JWT token, OidcAuthenticator authenticator, HttpServletRequest request) {
        Optional<String> maybeRefreshToken;
        boolean needsToBeRefreshed;
        boolean bl = needsToBeRefreshed = TokenUtils.hasMatchingIssuer(token, authenticator.tokenValidator.getIssuer()) && TokenUtils.expiresWithin(token, 300000L);
        if (needsToBeRefreshed && (maybeRefreshToken = authenticator.findRefreshToken(request)).isPresent() && authenticator.config.refreshUrl != null) {
            try {
                return Optional.of(this.tokenRefreshClient.refreshIdToken(authenticator.config.refreshUrl, maybeRefreshToken.get()));
            }
            catch (Exception e) {
                logger.error("Unable to refresh id token", (Throwable)e);
                return Optional.empty();
            }
        }
        return Optional.empty();
    }

    public void destroy() {
    }
}

