/*
 * Decompiled with CFR 0.152.
 */
package one.microstream.storage.types;

import java.nio.ByteBuffer;
import one.microstream.X;
import one.microstream.afs.types.AFile;
import one.microstream.afs.types.AFileSystem;
import one.microstream.chars.VarString;
import one.microstream.chars.XChars;
import one.microstream.collections.ArrayView;
import one.microstream.collections.XArrays;
import one.microstream.concurrency.XThreads;
import one.microstream.memory.XMemory;
import one.microstream.storage.exceptions.StorageException;
import one.microstream.storage.exceptions.StorageExceptionInitialization;
import one.microstream.storage.types.StorageClosableFile;
import one.microstream.storage.types.StorageLiveFileProvider;
import one.microstream.storage.types.StorageLockFile;
import one.microstream.storage.types.StorageLockFileSetup;
import one.microstream.storage.types.StorageOperationController;

public interface StorageLockFileManager
extends Runnable {
    default public StorageLockFileManager start() {
        this.setRunning(true);
        return this;
    }

    default public StorageLockFileManager stop() {
        this.setRunning(false);
        return this;
    }

    public boolean isRunning();

    public StorageLockFileManager setRunning(boolean var1);

    public static StorageLockFileManager New(StorageLockFileSetup setup, StorageOperationController operationController) {
        return new Default((StorageLockFileSetup)X.notNull((Object)setup), (StorageOperationController)X.notNull((Object)operationController));
    }

    public static Creator Creator() {
        return new Creator.Default();
    }

    public static interface Creator {
        public StorageLockFileManager createLockFileManager(StorageLockFileSetup var1, StorageOperationController var2);

        public static final class Default
        implements Creator {
            Default() {
            }

            @Override
            public StorageLockFileManager createLockFileManager(StorageLockFileSetup setup, StorageOperationController operationController) {
                return StorageLockFileManager.New(setup, operationController);
            }
        }
    }

    public static final class Default
    implements StorageLockFileManager {
        private final StorageLockFileSetup setup;
        private final StorageOperationController operationController;
        private transient boolean isRunning;
        private transient StorageLockFile lockFile;
        private transient LockFileData lockFileData;
        private transient ByteBuffer[] wrappedByteBuffer;
        private transient ArrayView<ByteBuffer> wrappedWrappedByteBuffer;
        private transient ByteBuffer directByteBuffer;
        private transient byte[] stringReadBuffer;
        private transient byte[] stringWriteBuffer;
        private transient VarString vs;
        private transient AFileSystem fileSystem;

        Default(StorageLockFileSetup setup, StorageOperationController operationController) {
            this.setup = setup;
            this.fileSystem = setup.lockFileProvider().fileSystem();
            this.operationController = operationController;
            this.vs = VarString.New();
            this.wrappedByteBuffer = new ByteBuffer[1];
            this.wrappedWrappedByteBuffer = X.ArrayView((Object[])this.wrappedByteBuffer);
            this.stringReadBuffer = new byte[64];
            this.stringWriteBuffer = (byte[])this.stringReadBuffer.clone();
            this.allocateBuffer(this.stringReadBuffer.length);
        }

        @Override
        public final synchronized boolean isRunning() {
            return this.isRunning;
        }

        @Override
        public final synchronized StorageLockFileManager setRunning(boolean running) {
            this.isRunning = running;
            return this;
        }

        private synchronized boolean checkIsRunning() {
            return this.isRunning && this.operationController.checkProcessingEnabled();
        }

        @Override
        public Default start() {
            this.ensureInitialized();
            StorageLockFileManager.super.start();
            return this;
        }

        @Override
        public final void run() {
            long updateInterval = this.setup.updateInterval();
            Exception closingCause = null;
            try {
                try {
                    this.checkInitialized();
                    while (this.checkIsRunning()) {
                        XThreads.sleep((long)updateInterval);
                        this.updateFile();
                    }
                }
                catch (Exception e) {
                    closingCause = e;
                    this.operationController.registerDisruption(e);
                    throw e;
                }
            }
            finally {
                this.ensureClosedLockFile(closingCause);
            }
        }

        private void ensureInitialized() {
            if (this.lockFile != null) {
                return;
            }
            try {
                this.initialize();
            }
            catch (Exception e) {
                this.operationController.registerDisruption(e);
                this.ensureClosedLockFile(e);
                throw e;
            }
        }

        private void checkInitialized() {
            if (this.lockFile != null) {
                return;
            }
            throw new StorageExceptionInitialization(String.valueOf(StorageLockFileManager.class.getSimpleName()) + " not initialized.");
        }

        private ByteBuffer ensureReadingBuffer(int fileLength) {
            this.ensureBufferCapacity(fileLength);
            if (this.stringReadBuffer.length != fileLength) {
                this.stringReadBuffer = new byte[fileLength];
            }
            this.directByteBuffer.limit(fileLength);
            return this.directByteBuffer;
        }

        private ArrayView<ByteBuffer> ensureWritingBuffer(byte[] bytes) {
            this.ensureBufferCapacity(bytes.length);
            this.directByteBuffer.limit(bytes.length);
            this.stringWriteBuffer = bytes;
            return this.wrappedWrappedByteBuffer;
        }

        private boolean ensureBufferCapacity(int capacity) {
            if (this.directByteBuffer.capacity() >= capacity) {
                return false;
            }
            XMemory.deallocateDirectByteBuffer((ByteBuffer)this.directByteBuffer);
            this.allocateBuffer(capacity);
            return true;
        }

        private void allocateBuffer(int capacity) {
            this.wrappedByteBuffer[0] = this.directByteBuffer = XMemory.allocateDirectNative((int)capacity);
        }

        private String readString() {
            this.fillReadBufferFromFile();
            return new String(this.stringReadBuffer, this.setup.charset());
        }

        private void fillReadBufferFromFile() {
            int fileLength = X.checkArrayRange((long)this.lockFile.size());
            this.lockFile.readBytes(this.ensureReadingBuffer(fileLength), 0L, (long)fileLength);
            XMemory.copyRangeToArray((long)XMemory.getDirectByteBufferAddress((ByteBuffer)this.directByteBuffer), (byte[])this.stringReadBuffer);
        }

        private LockFileData readLockFileData() {
            String currentFileData = this.readString();
            char[] chars = currentFileData.toCharArray();
            int sep1Index = Default.indexOfFirstNonNumberCharacter(chars, 0);
            int sep2Index = Default.indexOfFirstNonNumberCharacter(chars, sep1Index + 1);
            long currentTime = XChars.parse_longDecimal((char[])chars, (int)0, (int)sep1Index);
            long expirationTime = XChars.parse_longDecimal((char[])chars, (int)(sep1Index + 1), (int)(sep2Index - sep1Index - 1));
            String identifier = String.valueOf(chars, sep2Index + 1, chars.length - sep2Index - 1);
            return new LockFileData(currentTime, expirationTime, identifier);
        }

        static final int indexOfFirstNonNumberCharacter(char[] data, int offset) {
            int i = offset;
            while (i < data.length) {
                if (data[i] < '0' || data[i] > '9') {
                    return i;
                }
                ++i;
            }
            throw new StorageException("No separator found in lock file string.");
        }

        private boolean lockFileHasContent() {
            return this.lockFile.exists() && this.lockFile.size() > 0L;
        }

        private void initialize() {
            LockFileData initialFileData;
            StorageLiveFileProvider fileProvider = this.setup.lockFileProvider();
            AFile lockFile = fileProvider.provideLockFile();
            this.lockFile = StorageLockFile.New(lockFile);
            LockFileData lockFileData = initialFileData = this.lockFileHasContent() ? this.validateExistingLockFileData(true) : null;
            if (this.isReadOnlyMode()) {
                if (initialFileData != null) {
                    this.setToWriteBuffer(initialFileData);
                }
                return;
            }
            this.lockFileData = new LockFileData(this.setup.processIdentity(), this.setup.updateInterval());
            this.writeLockFileData();
        }

        private LockFileData validateExistingLockFileData(boolean firstAttempt) {
            LockFileData existingFiledata = this.readLockFileData();
            String identifier = this.setup.processIdentity();
            if (identifier.equals(existingFiledata.identifier)) {
                return existingFiledata;
            }
            if (existingFiledata.isLongExpired()) {
                return existingFiledata;
            }
            if (firstAttempt) {
                XThreads.sleep((long)existingFiledata.updateInterval);
                return this.validateExistingLockFileData(false);
            }
            throw new StorageException("Storage already in use by: " + existingFiledata.identifier);
        }

        private void checkForModifiedLockFile() {
            if (this.isReadOnlyMode() && !this.lockFileHasContent()) {
                return;
            }
            this.fillReadBufferFromFile();
            if (XArrays.equals((byte[])this.stringReadBuffer, (byte[])this.stringWriteBuffer, (int)this.stringWriteBuffer.length)) {
                return;
            }
            throw new StorageException("Concurrent lock file modification detected.");
        }

        private boolean isReadOnlyMode() {
            return !this.fileSystem.isWritable();
        }

        private void writeLockFileData() {
            if (this.isReadOnlyMode()) {
                return;
            }
            this.lockFileData.update();
            ArrayView<ByteBuffer> bb = this.setToWriteBuffer(this.lockFileData);
            this.lockFile.writeBytes((Iterable<? extends ByteBuffer>)bb);
        }

        private ArrayView<ByteBuffer> setToWriteBuffer(LockFileData lockFileData) {
            this.vs.reset().add(lockFileData.lastWriteTime).add(';').add(lockFileData.expirationTime).add(';').add(lockFileData.identifier);
            byte[] bytes = this.vs.encodeBy(this.setup.charset());
            ArrayView<ByteBuffer> bb = this.ensureWritingBuffer(bytes);
            XMemory.copyArrayToAddress((byte[])bytes, (long)XMemory.getDirectByteBufferAddress((ByteBuffer)this.directByteBuffer));
            return bb;
        }

        private void updateFile() {
            if (!this.checkIsRunning()) {
                return;
            }
            this.checkForModifiedLockFile();
            this.writeLockFileData();
        }

        private void ensureClosedLockFile(Throwable cause) {
            if (this.lockFile == null) {
                return;
            }
            StorageClosableFile.close(this.lockFile, cause);
            this.lockFile = null;
        }

        static final class LockFileData {
            long lastWriteTime;
            long expirationTime;
            final String identifier;
            final long updateInterval;

            LockFileData(String identifier, long updateInterval) {
                this.identifier = identifier;
                this.updateInterval = updateInterval;
            }

            LockFileData(long lastWriteTime, long expirationTime, String identifier) {
                this(identifier, LockFileData.deriveUpdateInterval(lastWriteTime, expirationTime));
                this.lastWriteTime = lastWriteTime;
                this.expirationTime = expirationTime;
            }

            final void update() {
                this.lastWriteTime = System.currentTimeMillis();
                this.expirationTime = this.lastWriteTime + this.updateInterval;
            }

            private static long deriveUpdateInterval(long lastWriteTime, long expirationTime) {
                long derivedInterval = expirationTime - lastWriteTime;
                if (derivedInterval <= 0L) {
                    throw new StorageException("Invalid lockfile timestamps: lastWriteTime = " + lastWriteTime + ", expirationTime = " + expirationTime);
                }
                return derivedInterval;
            }

            final boolean isLongExpired() {
                return System.currentTimeMillis() > this.expirationTime + this.updateInterval;
            }
        }
    }
}

