/*
 * Copyright 2004 Carlos Sanchez.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sf.oness.common.model.bo;

import java.beans.PropertyDescriptor;
import java.util.Collection;
import java.util.Map;

import net.sf.oness.common.model.auditing.AbstractAuditableObject;
import net.sf.oness.common.model.temporal.DateRange;
import net.sf.oness.common.model.util.CollectionUtils;

import org.apache.commons.beanutils.PropertyUtils;

/**
 * Basic business object. Some superclass methods are overriden only to set
 * xdoclet attributes.
 * 
 * @author Carlos Sanchez
 * @version $Revision: 1.6 $
 */
public abstract class AbstractBusinessObject extends AbstractAuditableObject
        implements AuditableBusinessObject {

    private String comments;

    /**
     * @hibernate.id generator-class="increment"
     * 
     * @return
     */
    public Long getId() {
        return super.getId();
    }

    /**
     * @hibernate.property not-null="false"
     * 
     * @return Returns the code.
     */
    public Long getCode() {
        return super.getCode();
    }

    /**
     * @hibernate.property type="net.sf.oness.common.model.temporal.hibernate.DateRangeType"
     * @hibernate.column name="ttstart" not-null="true"
     * @hibernate.column name="ttend"
     * 
     * @see net.sf.oness.common.model.auditing.Auditable#getTransactionTime()
     */
    public DateRange getTransactionTime() {
        return super.getTransactionTime();
    }

    /**
     * @hibernate.property not-null="false"
     * 
     * @see net.sf.oness.common.model.auditing.Auditable#getCreatedBy()
     */
    public String getCreatedBy() {
        return super.getCreatedBy();
    }

    /**
     * @hibernate.property not-null="false"
     * 
     * @see net.sf.oness.common.model.auditing.Auditable#getDeletedBy()
     */
    public String getDeletedBy() {
        return super.getDeletedBy();
    }

    /**
     * Comments for this object
     * 
     * @hibernate.property not-null="false"
     * 
     * @return The comments
     */
    public String getComments() {
        return comments;
    }

    public void setComments(String comments) {
        this.comments = comments;
    }

    /**
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object o) {
        return CollectionIgnoringEqualsBuilder.reflectionEquals(this, o);
    }

    /**
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        return CollectionIgnoringHashCodeBuilder.reflectionHashCode(this);
    }

    /**
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return new CollectionIgnoringReflectionToStringBuilder(this).toString();
    }

    /**
     * This implementation calls super.clone and then clones collections without
     * cloning its elements. Returns <code>null</code> on any exception.
     * 
     * @see java.lang.Object#clone()
     */
    public Object clone() {
        try {
            Object o = super.clone();
            PropertyDescriptor origDescriptors[] = PropertyUtils
                    .getPropertyDescriptors(this);
            for (int i = 0; i < origDescriptors.length; i++) {
                Class clazz = origDescriptors[i].getPropertyType();
                if (CollectionUtils.isCollection(clazz)) {
                    /* if it's a Collection */
                    String name = origDescriptors[i].getName();
                    Collection old = (Collection) PropertyUtils
                            .getSimpleProperty(this, name);
                    PropertyUtils.setSimpleProperty(o, name, CollectionUtils
                            .clone(old));
                }
                if (CollectionUtils.isMap(clazz)) {
                    /* if it's a Collection */
                    String name = origDescriptors[i].getName();
                    Map old = (Map) PropertyUtils.getSimpleProperty(this, name);
                    PropertyUtils.setSimpleProperty(o, name, CollectionUtils
                            .clone(old));
                }
            }
            return o;
        } catch (Exception e) {
            log.error("Exception cloning bean: " + this + "\n"
                    + "Exception was " + e + ": " + e.getLocalizedMessage());
            return null;
        }

    }
}