package org.accidia.dbz.impl;

import com.google.common.base.Strings;
import org.accidia.dbz.IDbz;
import com.sleepycat.bind.EntryBinding;
import com.sleepycat.bind.serial.ClassCatalog;
import com.sleepycat.bind.serial.SerialBinding;
import com.sleepycat.bind.serial.StoredClassCatalog;
import com.sleepycat.collections.StoredMap;
import com.sleepycat.je.*;

import java.io.File;
import java.io.IOException;

import static com.google.common.base.Preconditions.checkArgument;

public class DbzOnBDBImpl implements IDbz {
    private final EnvironmentConfig environmentConfig;
    private final Environment environment;
    private final DatabaseConfig databaseConfig;
    private final Database database;
    private final ClassCatalog classCatalog;
    private final EntryBinding<String> keyBinding;
    private final EntryBinding<byte[]> valueBinding;

    private final StoredMap<String, byte[]> storedMap;

    public DbzOnBDBImpl(final String homeDirectory, final String databaseName) throws DatabaseException {
        this.environmentConfig = new EnvironmentConfig();
        this.environmentConfig.setAllowCreate(true);
        this.environmentConfig.setTransactional(true);
        this.environment = new Environment(new File(homeDirectory), environmentConfig);
        this.databaseConfig = new DatabaseConfig();
        this.databaseConfig.setTransactional(true);
        this.databaseConfig.setAllowCreate(true);
        this.database = this.environment.openDatabase(null, databaseName, this.databaseConfig);
        this.classCatalog = new StoredClassCatalog(this.database);
        this.keyBinding = new SerialBinding<>(this.classCatalog, String.class);
        this.valueBinding = new SerialBinding<>(this.classCatalog, byte[].class);

        this.storedMap = new StoredMap<>(this.database, this.keyBinding, this.valueBinding, true);
    }
    @Override
    public byte[] get(final String guid) throws IOException {
        checkArgument(!Strings.isNullOrEmpty(guid), "null/empty guid");

        return getStoredMap().get(guid);
    }

    @Override
    public void set(final String guid, final byte[] value) throws IOException {
        checkArgument(!Strings.isNullOrEmpty(guid), "null/empty guid");
        checkArgument(value != null, "null value");

        getStoredMap().put(guid, value);
    }

    @Override
    public void delete(final String guid) throws IOException {
        checkArgument(!Strings.isNullOrEmpty(guid), "null/empty guid");

        getStoredMap().remove(guid);
    }

    public StoredMap<String, byte[]> getStoredMap() {
        return this.storedMap;
    }

    public void close() throws DatabaseException {
        this.database.close();
        this.classCatalog.close();
        this.environment.close();
    }
}

