package org.accidia.dbz.impl;

import com.google.common.base.Strings;
import com.google.inject.Inject;
import org.accidia.dbz.IDbz;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.support.JdbcDaoSupport;

import javax.sql.DataSource;
import java.io.IOException;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import static com.google.common.base.Preconditions.checkArgument;

public class DbzOnDerbyImpl extends JdbcDaoSupport implements IDbz {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Inject
    public DbzOnDerbyImpl(final DataSource dataSource) {
        checkArgument(dataSource != null, "null dataSource");
        setDataSource(dataSource);
        createTableIfNotExists();
    }

    private void createTableIfNotExists() {
        logger.info("trying to create the table if not exists");
        try {
            final String sql = "CREATE TABLE KEYVALUE " +
                    "(GUID varchar(128) NOT NULL PRIMARY KEY,  " +
                    "VALUEBYTES blob(16M)  " +
                    ")";
            getJdbcTemplate().update(sql);
        } catch (Exception e) {
            logger.error("exception while trying to create table", e);
        }
    }

    @SuppressWarnings("unchecked")
    private final RowMapper genericProtobufRowMapper = new KeyValueRowMapper();

    @Override
    public byte[] get(final String guid) throws IOException {
        checkArgument(!Strings.isNullOrEmpty(guid), "null/empty guid");

        final String sql = "SELECT GUID,VALUEBYTES FROM KEYVALUE WHERE GUID=?";
        final List list = getJdbcTemplate().query(sql, new Object[]{guid}, this.genericProtobufRowMapper);
        if (list == null || list.isEmpty()) {
            return null;
        }
        if (list.size() == 1) {
            return (byte[]) list.get(0);
        }
        throw new AssertionError("should never reach here");
    }

    @Override
    public void set(final String guid, final byte[] value) throws IOException {
        checkArgument(!Strings.isNullOrEmpty(guid), "null/empty guid");
        checkArgument(value != null, "null value");

        // write to database
        final String sql = "INSERT INTO KEYVALUE (GUID,VALUEBYTES) "
                + "VALUES (?,?) ";
        getJdbcTemplate().update(sql, new Object[]{guid, value});
    }

    @Override
    public void delete(final String guid) throws IOException {
        checkArgument(!Strings.isNullOrEmpty(guid), "null/empty guid");
        // write to database
        final String sql = "DELETE FROM KEYVALUE WHERE GUID = ?";
        getJdbcTemplate().update(sql, new Object[]{guid});
    }

    @Override
    protected void finalize() throws Throwable {
        logger.info("finalizing derbydb");
        super.finalize();
        DriverManager.getConnection("jdbc:derby:;shutdown=true");
    }

    public static class KeyValueRowMapper implements RowMapper {
        private static final Logger logger = LoggerFactory.getLogger(KeyValueRowMapper.class);

        @Override
        @SuppressWarnings("unchecked")
        public byte[] mapRow(final ResultSet resultSet, final int rowIgnored) throws SQLException {
            try {
                return resultSet.getBytes("VALUEBYTES");
            } catch (SQLException e) {
                logger.warn("caught exception mapping row to object", e);
                throw new RuntimeException(e);
            }
        }
    }
}
