package org.accidia.echo.resources;

import com.codahale.metrics.annotation.ExceptionMetered;
import com.codahale.metrics.annotation.Metered;
import com.google.common.base.Strings;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.protobuf.ExtensionRegistry;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import com.googlecode.protobuf.format.JsonFormat;
import org.accidia.echo.EchoContext;
import org.accidia.echo.Constants;
import org.accidia.echo.protos.Protos;
import org.accidia.echo.services.IObjectsService;
import org.accidia.echo.services.ITenantService;
import org.accidia.echo.protoserver.misc.AsyncResponses;
import org.accidia.echo.protoserver.misc.MediaTypes;
import org.glassfish.jersey.server.ManagedAsync;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.*;
import javax.ws.rs.container.AsyncResponse;
import javax.ws.rs.container.Suspended;
import javax.ws.rs.core.MediaType;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import static com.google.common.base.Preconditions.checkArgument;

@Path("v1/object/")
public class ObjectResource {
    private final Logger logger = LoggerFactory.getLogger(getClass());
    private final long requestTimeoutInSeconds = EchoContext.INSTANCE
            .getConfiguration().getConfig().getLong(Constants.CONFIG_KEY__RESOURCE_TIMEOUT_SECONDS);

    private final ITenantService tenantService = EchoContext.INSTANCE.getInjector()
            .getInstance(ITenantService.class);
    private final ExtensionRegistry extensionRegistry = ExtensionRegistry.newInstance();

    @GET
    @Metered @ExceptionMetered
    @Path("/{tenant}")
    @Produces({MediaTypes.APPLICATION_PROTOBUF + ";qs=.5", MediaType.APPLICATION_JSON})
    @ManagedAsync
    public void getObjectList(@Suspended final AsyncResponse asyncResponse,
                              @PathParam("tenant") final String tenant,
                              @DefaultValue("0") @QueryParam("start") final String startString,
                              @DefaultValue("-1") @QueryParam("count") final String countString) {
        logger.debug("getObjects()");

        // first validate parameters
        this.tenantService.validateTenant(tenant);
        final int start, count;
        try {
            start = Integer.valueOf(startString);
            count = Integer.valueOf(countString);
        } catch (final NumberFormatException e) {
            throw new IllegalArgumentException(e);
        }
        checkArgument(count >= -1, "invalid count");
        checkArgument(start >= 0, "invalid start");

        // then add timeout and completion handler
        AsyncResponses.addTimeoutHandler(asyncResponse, this.requestTimeoutInSeconds, TimeUnit.SECONDS);
        AsyncResponses.addCompletionCallback(asyncResponse);

        // then do get the object
        doGetObjectList(asyncResponse, tenant, start, count);
    }

    @GET
    @Metered
    @ExceptionMetered
    @Path("/{tenant}/{csvkeys}")
    @Produces({MediaTypes.APPLICATION_PROTOBUF + ";qs=.5"})
    @ManagedAsync
    public void getObjectAsProtobuf(@Suspended final AsyncResponse asyncResponse,
                                    @PathParam("tenant") final String tenant,
                                    @PathParam("csvkeys") final String csvKeys,
                                    @DefaultValue("ts") @QueryParam("orderby") final String orderby) {
        logger.debug("getObjectAsProtobuf()");

        // first validate parameters
        this.tenantService.validateTenant(tenant);
        checkArgument(!Strings.isNullOrEmpty(csvKeys.trim()), "null/empty key");

        // then add timeout and completion handler
        AsyncResponses.addTimeoutHandler(asyncResponse, this.requestTimeoutInSeconds, TimeUnit.SECONDS);
        AsyncResponses.addCompletionCallback(asyncResponse);

        // then do get the object
        if (!csvKeys.contains(",")) {
            // only one key
            doGetObject(asyncResponse, tenant, csvKeys);
            return;
        }
        doGetObjects(asyncResponse, tenant, Arrays.asList(csvKeys.split(",")));
    }

    @GET
    @Metered
    @ExceptionMetered
    @Path("/{tenant}/{csvkeys}")
    @Produces({MediaType.APPLICATION_JSON})
    @ManagedAsync
    public void getObjectAsJson(@Suspended final AsyncResponse asyncResponse,
                                @PathParam("tenant") final String tenant,
                                @PathParam("csvkeys") final String csvKeys) {
        logger.debug("getObjectAsJson()");

        // first validate parameters
        this.tenantService.validateTenant(tenant);
        checkArgument(!Strings.isNullOrEmpty(csvKeys.trim()), "null/empty key");

        // then add timeout and completion handler
        AsyncResponses.addTimeoutHandler(asyncResponse, this.requestTimeoutInSeconds, TimeUnit.SECONDS);
        AsyncResponses.addCompletionCallback(asyncResponse);

        // then do get the object
        if (!csvKeys.contains(",")) {
            // only one key
            doGetObject(asyncResponse, tenant, csvKeys);
            return;
        }
        doGetObjectsAsJson(asyncResponse, tenant, Arrays.asList(csvKeys.split(",")));
    }

    @GET
    @Metered
    @ExceptionMetered
    @Path("/{tenant}/{csvkeys}/{csvfields}")
    @Produces({MediaTypes.APPLICATION_PROTOBUF + ";qs=0.5", MediaType.APPLICATION_JSON})
    @ManagedAsync
    public void getPartialObject(@Suspended final AsyncResponse asyncResponse,
                                 @PathParam("tenant") final String tenant,
                                 @PathParam("csvkeys") final String csvKeys,
                                 @PathParam("csvfields") final String csvFields) {
        logger.debug("getPartialObject()");

        // validate
        this.tenantService.validateTenant(tenant);
        checkArgument(!Strings.isNullOrEmpty(csvKeys.trim()), "null/empty key");
        checkArgument(!Strings.isNullOrEmpty(csvFields.trim()), "null/empty fields");

        // timeout and completion handler
        AsyncResponses.addTimeoutHandler(asyncResponse, this.requestTimeoutInSeconds, TimeUnit.SECONDS);
        AsyncResponses.addCompletionCallback(asyncResponse);

        // do get
        if (!csvKeys.contains(",")) {
            doGetPartialObject(asyncResponse, tenant, csvKeys, csvFields);
        }
        doGetPartialObjects(asyncResponse, tenant, Arrays.asList(csvKeys.split(",")), csvFields);
    }

    @POST
    @Metered
    @ExceptionMetered
    @Path("/{tenant}/{key}")
    @Consumes({MediaType.APPLICATION_JSON})
    @ManagedAsync
    public void postJsonObject(@Suspended final AsyncResponse asyncResponse,
                           @PathParam("tenant") final String tenant,
                           @PathParam("key") final String key,
                           final String jsonObject) {
        logger.debug("postObject()");

        // validate
        this.tenantService.validateTenant(tenant);
        checkArgument(!Strings.isNullOrEmpty(key.trim()), "null/empty key");
        checkArgument(!Strings.isNullOrEmpty(jsonObject.trim()), "null/empty json object");

        // timeout and completion handler
        AsyncResponses.addTimeoutHandler(asyncResponse, this.requestTimeoutInSeconds, TimeUnit.SECONDS);
        AsyncResponses.addCompletionCallback(asyncResponse);

        // json to object
        final Message.Builder builder = this.tenantService.getTenant(tenant)
                .getDefaultInstance().newBuilderForType();
        try {
            JsonFormat.merge(jsonObject, this.extensionRegistry, builder);
        } catch (JsonFormat.ParseException | RuntimeException e) {
            throw new IllegalArgumentException("invalid json", e);
        }

        // do post
        doPostObject(asyncResponse, tenant, key, builder.buildPartial());
    }

    @POST
    @Metered
    @ExceptionMetered
    @Path("/{tenant}/{key}")
    @Consumes({MediaTypes.APPLICATION_PROTOBUF})
    @ManagedAsync
    public void postProtobufObject(@Suspended final AsyncResponse asyncResponse,
                           @PathParam("tenant") final String tenant,
                           @PathParam("key") final String key,
                           final byte[] objectBytes) {
        logger.debug("postObject()");

        // validate
        this.tenantService.validateTenant(tenant);
        checkArgument(!Strings.isNullOrEmpty(key.trim()), "null/empty key");
        checkArgument(objectBytes != null && objectBytes.length > 0, "null/empty object bytes");

        // timeout and completion handler
        AsyncResponses.addTimeoutHandler(asyncResponse, this.requestTimeoutInSeconds, TimeUnit.SECONDS);
        AsyncResponses.addCompletionCallback(asyncResponse);

        // bytes to object
        final Message.Builder builder = this.tenantService.getTenant(tenant)
                .getDefaultInstance().newBuilderForType();
        try {
            builder.mergeFrom(objectBytes);
        } catch (final InvalidProtocolBufferException e) {
            throw new IllegalArgumentException("invalid protobuf object", e);
        }

        // do post
        doPostObject(asyncResponse, tenant, key, builder.buildPartial());
    }

    protected IObjectsService getObjectServicesForTenant(final String tenant) {
        final IObjectsService objectsServices = this.tenantService.getObjectsServicesForTenant(tenant);
        checkArgument(objectsServices != null, "invalid tenant: " + tenant);
        return objectsServices;
    }

    protected void doGetObjectList(final AsyncResponse asyncResponse,
                                   final String tenant,
                                   final int start,
                                   final int count) {
        AsyncResponses.addCallbackForListenableFuture(asyncResponse,
                getObjectServicesForTenant(tenant).getObjectList("TODO", start, count));
    }

    protected void doGetObject(final AsyncResponse asyncResponse,
                               final String tenant,
                               final String key) {
        AsyncResponses.addCallbackForListenableFuture(asyncResponse,
                getObjectServicesForTenant(tenant).getObject(key));
    }


    protected void doGetObjects(final AsyncResponse asyncResponse,
                                final String tenant,
                                final List<String> keys) {
        final ListenableFuture<Map<String, Message>> futureResult = getObjectServicesForTenant(tenant).getObjects(keys);
        Futures.addCallback(futureResult, new FutureCallback<Map<String, Message>>() {
            @Override
            public void onSuccess(final Map<String, Message> result) {
                checkArgument(result != null, "null result");
                final Protos.ListResult.Builder listResultBuilder = Protos.ListResult.newBuilder();
                for (final Map.Entry<String, Message> entry : result.entrySet()) {
                    listResultBuilder.addObjects(
                            Protos.ObjectResult.newBuilder()
                                    .setKey(entry.getKey())
                                    .setObject(entry.getValue().toByteString())
                                    .setClassName(entry.getValue().getClass().getName())
                    );
                }
                asyncResponse.resume(listResultBuilder.build());
            }

            @Override
            public void onFailure(final Throwable t) {
                logger.error("get objects failed: ", t);
                asyncResponse.resume(t);
            }
        });
    }

    protected void doGetObjectsAsJson(final AsyncResponse asyncResponse,
                                      final String tenant,
                                      final List<String> keys) {
        final ListenableFuture<Map<String, Message>> futureResult = getObjectServicesForTenant(tenant).getObjects(keys);
        Futures.addCallback(futureResult, new FutureCallback<Map<String, Message>>() {
            @Override
            public void onSuccess(final Map<String, Message> result) {
                checkArgument(result != null, "null result");
                asyncResponse.resume(result);
            }

            @Override
            public void onFailure(final Throwable t) {
                logger.error("get objects as json failed: ", t);
                asyncResponse.resume(t);
            }
        });
    }

    protected void doGetPartialObject(final AsyncResponse asyncResponse,
                                      final String tenant,
                                      final String key,
                                      final String csvFields) {

        AsyncResponses.addCallbackForListenableFuture(asyncResponse,
                getObjectServicesForTenant(tenant)
                        .getPartialObject(key, Arrays.asList(csvFields.split(","))));
    }

    protected void doGetPartialObjects(final AsyncResponse asyncResponse,
                                       final String tenant,
                                       final List<String> keys,
                                       final String csvFields) {

        final ListenableFuture<Map<String, Message>> futureResult = getObjectServicesForTenant(tenant)
                .getPartialObjects(keys, Arrays.asList(csvFields.split(",")));

        Futures.addCallback(futureResult, new FutureCallback<Map<String, Message>>() {
            @Override
            public void onSuccess(final Map<String, Message> result) {
                checkArgument(result != null, "null result");
                asyncResponse.resume(result);
            }

            @Override
            public void onFailure(final Throwable t) {
                logger.error("get objects fields failed: ", t);
                asyncResponse.resume(t);
            }
        });
    }

    protected void doPostObject(final AsyncResponse asyncResponse,
                                final String tenant,
                                final String key,
                                final Message object) {
        AsyncResponses.addCallbackForListenableFuture(asyncResponse,
                getObjectServicesForTenant(tenant)
                        .storeObject(key, object));
    }
}

