package org.accidia.echo.protobuf;

import com.google.common.base.Charsets;
import com.google.common.base.Strings;
import com.google.protobuf.DescriptorProtos;
import com.google.protobuf.Descriptors;
import com.google.protobuf.DynamicMessage;
import com.google.protobuf.Message;
import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.DefaultExecutor;
import org.apache.commons.exec.ExecuteWatchdog;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InvalidClassException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.protobuf.Descriptors.DescriptorValidationException;
import static com.google.protobuf.Descriptors.FieldDescriptor;

/**
 * Bunch of utility functions for protobuf objects
 */
public class Protobufs {
    private static final Logger logger = LoggerFactory.getLogger(Protobufs.class);

    /**
     * Given a protobuf message object, return the list of all fields names to upper case
     *
     * @param message protobuf object
     * @return list of field names in capital case
     */
    public static List<String> getAllFieldNamesToUpperCase(final Message message) {
        logger.debug("getAllFieldNamesToUpperCase(message)");

        // getAllFieldNames validates message
        final List<String> fieldNames = getAllFieldNames(message);
        final List<String> fieldNamesToUpperCase = new ArrayList<>(fieldNames.size());
        for (final String fieldName : fieldNames) {
            fieldNamesToUpperCase.add(fieldName.toUpperCase());
        }
        return fieldNamesToUpperCase;
    }

    /**
     * Given a protobuf message object, return the list of all field names
     *
     * @param message the protobuf object
     * @return list of field names
     */
    public static List<String> getAllFieldNames(final Message message) {
        logger.debug("getAllFieldNames(message)");
        checkArgument(message != null, "null message");

        // get the list of all fields from the message descriptor
        final List<FieldDescriptor> fields = message.getDescriptorForType().getFields();
        final List<String> fieldsList = new ArrayList<>(fields.size());
        // for each field, add the name to the result list
        for (final FieldDescriptor fieldDescriptor : fields) {
            fieldsList.add(fieldDescriptor.getName());
        }
        return fieldsList;
    }

    /**
     * Given a protobuf message object, return the list of fields names which have
     * value set in the object
     *
     * @param message the protobuf object
     * @return list of field names
     */
    public static List<String> getDefinedFieldNames(final Message message) {
        logger.debug("getDefinedFieldNames(message)");
        checkArgument(message != null, "null message");

        // get the list of defined (those with value) from message
        final Map<FieldDescriptor, Object> fieldsMap = message.getAllFields();
        if (fieldsMap == null || fieldsMap.isEmpty()) {
            return Collections.emptyList();
        }
        final List<String> fieldsList = new ArrayList<>(fieldsMap.size());
        // for each field, add the name to the result list
        for (final FieldDescriptor fieldDescriptor : fieldsMap.keySet()) {
            fieldsList.add(fieldDescriptor.getName());
        }
        return fieldsList;
    }

    public static Object getValueForFieldName(final Message message, final String fieldName) {
        logger.debug("getValueForFieldName(message,fieldName)");
        checkArgument(message != null, "null message");
        checkArgument(!Strings.isNullOrEmpty(fieldName), "null/empty fieldName");

        for (final FieldDescriptor fieldDescriptor : message.getDescriptorForType().getFields()) {
            if (fieldDescriptor.getName().equalsIgnoreCase(fieldName)) {
                return message.getField(fieldDescriptor);
            }
        }
        throw new IllegalArgumentException("invalid fieldName: " + fieldName);
    }

    /**
     * Given a protocol buffer message description as a string, and the message type name, returns
     * the default instance of the given message type.
     *
     * @param proto           protocol buffer message description as a string
     * @param messageTypeName the message type name to get the default instance for
     * @return default instance of type messageTypeName
     * @throws IOException
     * @throws DescriptorValidationException
     */
    public static Message getDefaultInstanceForProto(final String proto, final String messageTypeName)
            throws IOException, DescriptorValidationException, ReflectiveOperationException {
        logger.debug("getDefaultInstanceForProto(proto)");

        checkArgument(!Strings.isNullOrEmpty(proto), "null/empty proto");
        checkArgument(!Strings.isNullOrEmpty(messageTypeName), "null/empty messageTypeName");

        // create a temporary file and store the proto in it
        logger.info("creating temporary file to contain proto");
        final File protoFile = File.createTempFile("proto-", ".proto");
        protoFile.deleteOnExit();
        FileUtils.writeStringToFile(protoFile, proto, Charsets.UTF_8);

        // call `protoc` to compile the file
        final String outputDirectory = "/tmp/echo/protos/";
        final String mkdirCommand = "mkdir -p " + outputDirectory;

        // set up executor
        final DefaultExecutor executor = new DefaultExecutor();
        executor.setExitValue(0);
        final ExecuteWatchdog watchdog = new ExecuteWatchdog(3000); // 3 seconds to run
        executor.setWatchdog(watchdog);

        logger.info("executing command: {}", mkdirCommand);
        executor.execute(CommandLine.parse(mkdirCommand));

        // run protoc command to generate java files
        final String protocCommand = "protoc --java_out=" + outputDirectory
                + "  --descriptor_set_out=" + protoFile.getAbsolutePath().concat(".desc")
                + " --proto_path=" + protoFile.getParentFile().getAbsolutePath()
                + " " + protoFile.getAbsolutePath();

        final CommandLine protocCommandLine = CommandLine.parse(protocCommand);
        logger.info("executing command: {}", protocCommand);
        final int exitValue = executor.execute(protocCommandLine);
        if (exitValue != 0) {
            throw new RuntimeException("command returned non-zero: " + protocCommand);
        }

        // load .desc files to understand protobuf schema
        final DescriptorProtos.FileDescriptorSet fileDescriptorSet = DescriptorProtos.FileDescriptorSet.parseFrom(
                new FileInputStream(protoFile.getAbsolutePath().concat(".desc"))
        );

        // loop through all descriptors; if the given message type is found, return the default instance for it
        for (final DescriptorProtos.FileDescriptorProto fileDescriptorProto : fileDescriptorSet.getFileList()) {
            final Descriptors.FileDescriptor fileDescriptor = Descriptors.FileDescriptor
                    .buildFrom(fileDescriptorProto, new Descriptors.FileDescriptor[]{});

            for (final Descriptors.Descriptor descriptor : fileDescriptor.getMessageTypes()) {
                final String javaPackageName = fileDescriptorProto.getOptions().getJavaPackage();
                final String javaOuterClassName = fileDescriptorProto.getOptions().getJavaOuterClassname();
                final StringBuilder classNameBuilder = new StringBuilder();
                if (!Strings.isNullOrEmpty(javaPackageName)) {
                    classNameBuilder.append(javaPackageName).append(".");
                }
                if (!Strings.isNullOrEmpty(javaOuterClassName)) {
                    classNameBuilder.append(javaOuterClassName).append("$");
                }
                classNameBuilder.append(descriptor.getName());
                final String className = classNameBuilder.toString();

                logger.info("class name: {}", className);
                if (className.equalsIgnoreCase(messageTypeName)) {
                    logger.info("message type name found: {}", messageTypeName);
                    final DynamicMessage.Builder builder = DynamicMessage.newBuilder(descriptor);
                    return builder.buildPartial();
                }
            }
        }
        logger.warn("could not load message type: {} from protobuf description: {}", messageTypeName, proto);
        throw new InvalidClassException("invalid message type name: " + messageTypeName);
    }

}

