package org.accidia.echo.services;

import com.google.common.util.concurrent.ListenableFuture;
import com.google.protobuf.Message;

import java.util.List;
import java.util.Map;

/**
 * Objects Services provides functionality to work with Protobuf objects.
 */
public interface IObjectsService {

    /**
     * Find the object for the given tenant for the given key.
     *
     * @param key key for object
     * @return value associated to key in the given namespace
     */
    ListenableFuture<Message> getObject(final String key, boolean includeArchive);

    /**
     * Find all objects for the given set of keys.
     *
     * @param keysList list of keys
     * @return map of key to objects, one entry for each key in parameters
     */
    ListenableFuture<Map<String, Message>> getObjects(final List<String> keysList, boolean includeArchive);

    /**
     * Find an object and only return the specified fields.
     *
     * @param key    object key
     * @param fields fields to return
     * @return partial object with the given key
     */
    ListenableFuture<Message> getPartialObject(final String key, final List<String> fields, boolean includeArchive);

    /**
     * Get list of object keys
     *
     * @param start   starting index
     * @param count   number of objects
     * @param listKey key associated to the list
     * @return list of all objects
     */
    ListenableFuture<List<String>> getObjectList(final String listKey, int start, int count);

    /**
     * Get partial objects associated to the given set of keys.
     *
     * @param keys   list of keys
     * @param fields list of fields
     * @return map of key to partial objects, one for each key asked
     */
    ListenableFuture<Map<String, Message>> getPartialObjects(final List<String> keys, final List<String> fields, boolean includeArchive);

    /**
     * Store an object with the given key.
     *
     * @param key    the key
     * @param object the object to be stored
     * @return nothing really; exception is thrown if store fails, otherwise all silent
     */
    ListenableFuture<Message> storeObject(final String key, final Message object);

    /**
     * Store set of objects.
     *
     * @param keysToObjectsMap map of keys to objects to be stored
     * @return nothing really; if store fails, an exception is thrown, otherwise all silent
     */
    ListenableFuture<List<Message>> storeObjects(final Map<String, Message> keysToObjectsMap);
}

