package org.accidia.echo.services.impl;

import com.codahale.metrics.annotation.Timed;
import com.google.common.base.Strings;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.protobuf.Message;
import org.accidia.echo.dao.IProtobufDao;
import org.accidia.echo.services.IObjectsService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

import static com.google.common.base.Preconditions.checkArgument;

public class ObjectsService implements IObjectsService {
    private static final Logger logger = LoggerFactory.getLogger(ObjectsService.class);

    private final ListeningExecutorService listeningExecutorService;
    private final IProtobufDao protobufDao;

    public ObjectsService(final ListeningExecutorService listeningExecutorService,
                          final IProtobufDao protobufDao) {
        this.listeningExecutorService = listeningExecutorService;
        this.protobufDao = protobufDao;
    }

    @Override
    @Timed
    public ListenableFuture<Message> getObject(final String key, boolean includeArchive) {
        logger.debug("getObject(key)");
        checkArgument(!Strings.isNullOrEmpty(key), "null/empty key");
        return doGetObjectAsync(key, includeArchive);
    }

    @Override
    public ListenableFuture<List<String>> getObjectList(final String listKey, final int start, final int count) {
        logger.debug("getObjects(start,count)");
        checkArgument(start >= 0, "invalid start");
        checkArgument(count >= -1, "invalid count");
        return doGetObjectList(listKey, start, count);
    }

    @Override
    public ListenableFuture<Map<String, Message>> getObjects(final List<String> keysList, boolean includeArchive) {
        logger.debug("getObjects(keys)");
        checkArgument(keysList != null && !keysList.isEmpty(), "null/empty keys");
        return doGetObjects(keysList, includeArchive);
    }

    @Override
    public ListenableFuture<Message> getPartialObject(final String key, final List<String> fields, boolean includeArchive) {
        logger.debug("getPartialObject(key,fields)");
        checkArgument(!Strings.isNullOrEmpty(key), "null/empty key");

        // if fields are not given, return the whole object
        if (fields == null || fields.isEmpty()) {
            logger.debug("null or empty fields; returning the complete object");
            return doGetObjectAsync(key, includeArchive);
        }
        return doGetPartialObjectAsync(key, fields, includeArchive);
    }

    @Override
    public ListenableFuture<Map<String, Message>> getPartialObjects(final List<String> keys, final List<String> fields, boolean includeArchive) {
        logger.debug("getPartialObjects(keys,fields)");
        checkArgument(keys != null && !keys.isEmpty(), "null/empty keys");
        checkArgument(fields != null && !fields.isEmpty(), "null/empty fields");

        // if fields are not given, return the whole object
        if (fields == null || fields.isEmpty()) {
            logger.debug("null or empty fields; returning the complete object");
            return doGetObjects(keys, includeArchive);
        }
        return doGetPartialObjects(keys, fields, includeArchive);
    }

    @Override
    public ListenableFuture<Message> storeObject(final String key, final Message object) {
        logger.debug("storeObject(key,object)");
        checkArgument(!Strings.isNullOrEmpty(key), "null/empty key");
        checkArgument(object != null, "null object");

        return doStoreObjectAsync(key, object);
    }

    @Override
    public ListenableFuture<List<Message>> storeObjects(final Map<String, Message> keysToObjectsMap) {
        logger.debug("storeObjects(keysToObjectsMap)");
        checkArgument(keysToObjectsMap != null && !keysToObjectsMap.isEmpty(), "null/empty keysToObjectsMap");
        return doStoreObjects(keysToObjectsMap);
    }

    protected ListenableFuture<Message> doGetObjectAsync(final String key, boolean includeArchive) {
        return this.listeningExecutorService.submit(
                () -> this.protobufDao.findByKey(key, includeArchive)
        );
    }

    protected ListenableFuture<List<String>> doGetObjectList(final String listKey, final int start, final int count) {
        return this.listeningExecutorService.submit(
                () -> {
                    final int theCount = (count == -1) ? 100 : count;
                    return this.protobufDao.findList(listKey, start, theCount);
                }
        );
    }

    // parallelize gets for keys
    protected ListenableFuture<Map<String, Message>> doGetObjects(final List<String> keys, boolean includeArchive) {
        final List<ListenableFuture<Map.Entry<String, Message>>> futures = new ArrayList<>(keys.size());
        for (final String key : keys) {
            futures.add(transformMessageToObjectResult(key, doGetObjectAsync(key, includeArchive)));
        }
        return transformObjectResultsToListResult(Futures.allAsList(futures));
    }

    protected ListenableFuture<Message> doGetPartialObjectAsync(final String key, final List<String> fields, boolean includeArchive) {
        return this.listeningExecutorService.submit(
                () -> this.protobufDao.findFieldsByKey(key, fields, includeArchive)
        );
    }

    // parallelize gets for keys
    protected ListenableFuture<Map<String, Message>> doGetPartialObjects(final List<String> keys, final List<String> fields, boolean includeArchive) {
        final List<ListenableFuture<Map.Entry<String, Message>>> futures = new ArrayList<>(keys.size());
        for (final String key : keys) {
            futures.add(transformMessageToObjectResult(key, getPartialObject(key, fields, includeArchive)));
        }
        return transformObjectResultsToListResult(Futures.allAsList(futures));
    }

    protected ListenableFuture<Message> doStoreObjectAsync(final String key, final Message object) {
        return this.listeningExecutorService.submit(
                () -> {
                    doStoreObject(key, object);
                    // nothing is really returned
                    // if store is not successful, an exception will be thrown
                    return null;
                }
        );
    }

    protected void doStoreObject(final String key, final Message object) {
        try {
            this.protobufDao.store(key, object);
        } catch (final RuntimeException e) {
            // this is only for logging
            logger.warn("exception on storing object: ", e);
            throw e;
        }
    }

    // parallelize store object for keys
    protected ListenableFuture<List<Message>> doStoreObjects(final Map<String, Message> keysToObjectsMap) {
        final List<ListenableFuture<Message>> futures = new ArrayList<>(keysToObjectsMap.size());
        for (final Map.Entry<String, Message> objectEntry : keysToObjectsMap.entrySet()) {
            futures.add(doStoreObjectAsync(objectEntry.getKey(), objectEntry.getValue()));
        }
        return Futures.allAsList(futures);
    }

    // transform a list of listenable futures of object results to a listenable future of list result
    private ListenableFuture<Map<String, Message>> transformObjectResultsToListResult(
            final ListenableFuture<List<Map.Entry<String, Message>>> objectResults) {
        return Futures.transform(objectResults, (final List<Map.Entry<String, Message>> input) -> {
            final Map<String, Message> result = new HashMap<>(input.size());
            for (final Map.Entry<String, Message> entry : input) {
                result.put(entry.getKey(), entry.getValue());
            }
            return result;
        });
    }

    private ListenableFuture<Map.Entry<String, Message>> transformMessageToObjectResult(
            final String key,
            final ListenableFuture<Message> future) {
        return Futures.transform(future, (final Message object) -> new AbstractMap.SimpleEntry<>(key, object));
    }
}

