package org.accidia.echo.dao;

import com.google.protobuf.Message;

import java.util.List;
import java.util.Map;

public interface IProtobufDao {

    /**
     * Returns map of key to object for all objects in the given data source
     *
     * @return key to object map
     */
    List<Message> getAll();

    /**
     * Returns the object assigned to the given key
     *
     * @param key key assigned to the object to be fetched
     * @return the protobuf object assigned to the given key if found, null otherwise
     */
    Message findByKey(final String key);

    /**
     * Returns partial object containing only the specified fields
     *
     * @param key    key assigned to the object to be fetched
     * @param fields list of field names to be returned
     * @return partial message object
     */
    Message findFieldsByKey(final String key, final List<String> fields);

    /**
     * Returns list of objects starting from the given index; the list
     * will be maximum of size 'count'
     *
     * @param key     list key
     * @param start   the starting index
     * @param count   the maximum count of objects to return
     * @return list of object keys
     */
    List<String> findList(final String key, final int start, final int count);

    /**
     * Store the given object
     *
     * @param key    the key assigned to the object
     * @param object the protobuf object to be stored
     */
    void store(final String key, final Message object);

    /**
     * Add an object to a list
     *
     * @param listKey   key associated to the list
     * @param objectKey object key
     */
    void addToList(final String listKey, final String objectKey);

    /**
     * Store or update the given object
     *
     * @param key    the key assigned to the object
     * @param object the protobuf object to be stored
     */
    void storeOrUpdate(final String key, final Message object);

    /**
     * Mark the given key as archived.
     *
     * @param key the key referring to the object to be archived.
     */
    void archive(final String key);

    Message getMessageDefaultInstance();
}

