/*
 * Decompiled with CFR 0.152.
 */
package fc.cron;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.MutableDateTime;
import org.joda.time.ReadableInstant;

public class CronExpression {
    private final String expr;
    private final SimpleField secondField;
    private final SimpleField minuteField;
    private final SimpleField hourField;
    private final DayOfWeekField dayOfWeekField;
    private final SimpleField monthField;
    private final DayOfMonthField dayOfMonthField;

    public CronExpression(String expr) {
        this(expr, true);
    }

    public CronExpression(String expr, boolean withSeconds) {
        if (expr == null) {
            throw new IllegalArgumentException("expr is null");
        }
        this.expr = expr;
        int expectedParts = withSeconds ? 6 : 5;
        String[] parts = expr.split("\\s+");
        if (parts.length != expectedParts) {
            throw new IllegalArgumentException(String.format("Invalid cron expression [%s], expected %s felt, got %s", expr, expectedParts, parts.length));
        }
        int ix = withSeconds ? 1 : 0;
        this.secondField = new SimpleField(CronFieldType.SECOND, withSeconds ? parts[0] : "0");
        this.minuteField = new SimpleField(CronFieldType.MINUTE, parts[ix++]);
        this.hourField = new SimpleField(CronFieldType.HOUR, parts[ix++]);
        this.dayOfMonthField = new DayOfMonthField(parts[ix++]);
        this.monthField = new SimpleField(CronFieldType.MONTH, parts[ix++]);
        this.dayOfWeekField = new DayOfWeekField(parts[ix++]);
    }

    public static CronExpression create(String expr) {
        return new CronExpression(expr, true);
    }

    public static CronExpression createWithoutSeconds(String expr) {
        return new CronExpression(expr, false);
    }

    public DateTime nextTimeAfter(DateTime afterTime) {
        return this.nextTimeAfter(afterTime, afterTime.plusYears(4));
    }

    public DateTime nextTimeAfter(DateTime afterTime, long durationInMillis) {
        return this.nextTimeAfter(afterTime, afterTime.plus(durationInMillis));
    }

    public DateTime nextTimeAfter(DateTime afterTime, DateTime dateTimeBarrier) {
        MutableDateTime nextTime = new MutableDateTime((Object)afterTime);
        nextTime.setMillisOfSecond(0);
        nextTime.secondOfDay().add(1);
        while (true) {
            if (!this.secondField.matches(nextTime.getSecondOfMinute())) {
                nextTime.secondOfDay().add(1);
                continue;
            }
            if (!this.minuteField.matches(nextTime.getMinuteOfHour())) {
                nextTime.minuteOfDay().add(1);
                nextTime.secondOfMinute().set(0);
                continue;
            }
            if (!this.hourField.matches(nextTime.getHourOfDay())) {
                nextTime.hourOfDay().add(1);
                nextTime.minuteOfHour().set(0);
                nextTime.secondOfMinute().set(0);
                continue;
            }
            if (!this.dayOfMonthField.matches(new LocalDate((Object)nextTime))) {
                nextTime.addDays(1);
                nextTime.setTime(0, 0, 0, 0);
                CronExpression.checkIfDateTimeBarrierIsReached(nextTime, dateTimeBarrier);
                continue;
            }
            if (!this.monthField.matches(nextTime.getMonthOfYear())) {
                nextTime.addMonths(1);
                nextTime.setDayOfMonth(1);
                nextTime.setTime(0, 0, 0, 0);
                CronExpression.checkIfDateTimeBarrierIsReached(nextTime, dateTimeBarrier);
                continue;
            }
            if (this.dayOfWeekField.matches(new LocalDate((Object)nextTime))) break;
            nextTime.addDays(1);
            nextTime.setTime(0, 0, 0, 0);
            CronExpression.checkIfDateTimeBarrierIsReached(nextTime, dateTimeBarrier);
        }
        return nextTime.toDateTime();
    }

    private static void checkIfDateTimeBarrierIsReached(MutableDateTime nextTime, DateTime dateTimeBarrier) {
        if (nextTime.isAfter((ReadableInstant)dateTimeBarrier)) {
            throw new IllegalArgumentException("No next execution time could be determined that is before the limit of " + dateTimeBarrier);
        }
    }

    public String toString() {
        return this.getClass().getSimpleName() + "<" + this.expr + ">";
    }

    static class DayOfMonthField
    extends BasicField {
        DayOfMonthField(String fieldExpr) {
            super(CronFieldType.DAY_OF_MONTH, fieldExpr);
        }

        boolean matches(LocalDate dato) {
            for (FieldPart part : this.parts) {
                if ("L".equals(part.modifier)) {
                    return dato.getDayOfMonth() == dato.dayOfMonth().getMaximumValue() - (part.from == null ? 0 : part.from);
                }
                if ("W".equals(part.modifier)) {
                    if (dato.getDayOfWeek() > 5) continue;
                    if (dato.getDayOfMonth() == part.from.intValue()) {
                        return true;
                    }
                    if (dato.getDayOfWeek() == 5) {
                        return dato.plusDays(1).getDayOfMonth() == part.from.intValue();
                    }
                    if (dato.getDayOfWeek() != 1) continue;
                    return dato.minusDays(1).getDayOfMonth() == part.from.intValue();
                }
                if (!this.matches(dato.getDayOfMonth(), part)) continue;
                return true;
            }
            return false;
        }

        @Override
        protected void validatePart(FieldPart part) {
            if (part.modifier != null && Arrays.asList("L", "W", "?").indexOf(part.modifier) == -1) {
                throw new IllegalArgumentException(String.format("Invalid modifier [%s]", part.modifier));
            }
            if (part.incrementModifier != null && !"/".equals(part.incrementModifier)) {
                throw new IllegalArgumentException(String.format("Invalid increment modifier [%s]", part.incrementModifier));
            }
        }

        @Override
        protected boolean matches(int val, FieldPart part) {
            return "?".equals(part.modifier) || super.matches(val, part);
        }
    }

    static class DayOfWeekField
    extends BasicField {
        DayOfWeekField(String fieldExpr) {
            super(CronFieldType.DAY_OF_WEEK, fieldExpr);
        }

        boolean matches(LocalDate dato) {
            for (FieldPart part : this.parts) {
                if ("L".equals(part.modifier)) {
                    return dato.getDayOfWeek() == part.from.intValue() && dato.getDayOfMonth() > dato.dayOfMonth().getMaximumValue() - 7;
                }
                if ("#".equals(part.incrementModifier)) {
                    if (dato.getDayOfWeek() == part.from.intValue()) {
                        int num = dato.getDayOfMonth() / 7;
                        return part.increment == (dato.getDayOfMonth() % 7 == 0 ? num : num + 1);
                    }
                    return false;
                }
                if (!this.matches(dato.getDayOfWeek(), part)) continue;
                return true;
            }
            return false;
        }

        @Override
        protected Integer mapValue(String value) {
            return "0".equals(value) ? Integer.valueOf(7) : super.mapValue(value);
        }

        @Override
        protected boolean matches(int val, FieldPart part) {
            return "?".equals(part.modifier) || super.matches(val, part);
        }

        @Override
        protected void validatePart(FieldPart part) {
            if (part.modifier != null && Arrays.asList("L", "?").indexOf(part.modifier) == -1) {
                throw new IllegalArgumentException(String.format("Invalid modifier [%s]", part.modifier));
            }
            if (part.incrementModifier != null && Arrays.asList("/", "#").indexOf(part.incrementModifier) == -1) {
                throw new IllegalArgumentException(String.format("Invalid increment modifier [%s]", part.incrementModifier));
            }
        }
    }

    static class SimpleField
    extends BasicField {
        SimpleField(CronFieldType fieldType, String fieldExpr) {
            super(fieldType, fieldExpr);
        }

        public boolean matches(int val) {
            if (val >= this.fieldType.from && val <= this.fieldType.to) {
                for (FieldPart part : this.parts) {
                    if (!this.matches(val, part)) continue;
                    return true;
                }
            }
            return false;
        }
    }

    static abstract class BasicField {
        private static final Pattern CRON_FELT_REGEXP = Pattern.compile("(?:                                             # start of group 1\n   (?:(?<all>\\*)|(?<ignorer>\\?)|(?<last>L))  # globalt flag (L, ?, *)\n | (?<start>[0-9]{1,2}|[a-z]{3,3})              # or start number or symbol\n      (?:                                        # start of group 2\n         (?<mod>L|W)                             # modifier (L,W)\n       | -(?<end>[0-9]{1,2}|[a-z]{3,3})        # or end nummer or symbol (in range)\n      )?                                         # end of group 2\n)                                              # end of group 1\n(?:(?<inkmod>/|\\#)(?<ink>[0-9]{1,7}))?        # increment and increment modifier (/ or \\#)\n", 6);
        final CronFieldType fieldType;
        final List<FieldPart> parts = new ArrayList<FieldPart>();

        private BasicField(CronFieldType fieldType, String fieldExpr) {
            this.fieldType = fieldType;
            this.parse(fieldExpr);
        }

        private void parse(String fieldExpr) {
            String[] rangeParts;
            for (String rangePart : rangeParts = fieldExpr.split(",")) {
                Matcher m = CRON_FELT_REGEXP.matcher(rangePart);
                if (!m.matches()) {
                    throw new IllegalArgumentException("Invalid cron field '" + rangePart + "' for field [" + (Object)((Object)this.fieldType) + "]");
                }
                String startNummer = m.group("start");
                String modifier = m.group("mod");
                String sluttNummer = m.group("end");
                String inkrementModifier = m.group("inkmod");
                String inkrement = m.group("ink");
                FieldPart part = new FieldPart();
                part.increment = 999;
                if (startNummer != null) {
                    part.from = this.mapValue(startNummer);
                    part.modifier = modifier;
                    if (sluttNummer != null) {
                        part.to = this.mapValue(sluttNummer);
                        part.increment = 1;
                    } else if (inkrement != null) {
                        part.to = this.fieldType.to;
                    } else {
                        part.to = part.from;
                    }
                } else if (m.group("all") != null) {
                    part.from = this.fieldType.from;
                    part.to = this.fieldType.to;
                    part.increment = 1;
                } else if (m.group("ignorer") != null) {
                    part.modifier = m.group("ignorer");
                } else if (m.group("last") != null) {
                    part.modifier = m.group("last");
                } else {
                    throw new IllegalArgumentException("Invalid cron part: " + rangePart);
                }
                if (inkrement != null) {
                    part.incrementModifier = inkrementModifier;
                    part.increment = Integer.valueOf(inkrement);
                }
                this.validateRange(part);
                this.validatePart(part);
                this.parts.add(part);
            }
        }

        protected void validatePart(FieldPart part) {
            if (part.modifier != null) {
                throw new IllegalArgumentException(String.format("Invalid modifier [%s]", part.modifier));
            }
            if (part.incrementModifier != null && !"/".equals(part.incrementModifier)) {
                throw new IllegalArgumentException(String.format("Invalid increment modifier [%s]", part.incrementModifier));
            }
        }

        private void validateRange(FieldPart part) {
            if (part.from != null && part.from < this.fieldType.from || part.to != null && part.to > this.fieldType.to) {
                throw new IllegalArgumentException(String.format("Invalid interval [%s-%s], must be %s<=_<=%s", part.from, part.to, this.fieldType.from, this.fieldType.to));
            }
            if (part.from != null && part.to != null && part.from > part.to) {
                throw new IllegalArgumentException(String.format("Invalid interval [%s-%s].  Rolling periods are not supported (ex. 5-1, only 1-5) since this won't give a deterministic result. Must be %s<=_<=%s", part.from, part.to, this.fieldType.from, this.fieldType.to));
            }
        }

        protected Integer mapValue(String value) {
            Integer idx;
            if (this.fieldType.names != null && (idx = Integer.valueOf(this.fieldType.names.indexOf(value.toUpperCase(Locale.getDefault())))) >= 0) {
                return idx + 1;
            }
            return Integer.valueOf(value);
        }

        protected boolean matches(int val, FieldPart part) {
            return val >= part.from && val <= part.to && (val - part.from) % part.increment == 0;
        }
    }

    static class FieldPart {
        private Integer from;
        private Integer to;
        private Integer increment;
        private String modifier;
        private String incrementModifier;

        FieldPart() {
        }
    }

    static enum CronFieldType {
        SECOND(0, 59, null),
        MINUTE(0, 59, null),
        HOUR(0, 23, null),
        DAY_OF_MONTH(1, 31, null),
        MONTH(1, 12, Arrays.asList("JAN", "FEB", "MAR", "APR", "MAY", "JUN", "JUL", "AUG", "SEP", "OCT", "NOV", "DEC")),
        DAY_OF_WEEK(1, 7, Arrays.asList("MON", "TUE", "WED", "THU", "FRI", "SAT", "SUN"));

        final int from;
        final int to;
        final List<String> names;

        private CronFieldType(int from, int to, List<String> names) {
            this.from = from;
            this.to = to;
            this.names = names;
        }
    }
}

