/*
 * Copyright 2016 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in alluxio.shaded.client.com.liance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package alluxio.shaded.client.io.netty.channel.kqueue;

import alluxio.shaded.client.io.netty.buffer.ByteBuf;
import alluxio.shaded.client.io.netty.buffer.ByteBufAllocator;
import alluxio.shaded.client.io.netty.channel.ChannelConfig;
import alluxio.shaded.client.io.netty.channel.RecvByteBufAllocator;
import alluxio.shaded.client.io.netty.channel.unix.PreferredDirectByteBufAllocator;
import alluxio.shaded.client.io.netty.util.UncheckedBooleanSupplier;
import alluxio.shaded.client.io.netty.util.internal.ObjectUtil;

import static java.lang.Math.max;
import static java.lang.Math.min;

final class KQueueRecvByteAllocatorHandle implements RecvByteBufAllocator.ExtendedHandle {
    private final PreferredDirectByteBufAllocator preferredDirectByteBufAllocator =
            new PreferredDirectByteBufAllocator();
    private final RecvByteBufAllocator.ExtendedHandle delegate;

    private final UncheckedBooleanSupplier defaultMaybeMoreDataSupplier = new UncheckedBooleanSupplier() {
        @Override
        public boolean get() {
            return maybeMoreDataToRead();
        }
    };
    private boolean overrideGuess;
    private boolean readEOF;
    private long numberBytesPending;

    KQueueRecvByteAllocatorHandle(RecvByteBufAllocator.ExtendedHandle handle) {
        delegate = ObjectUtil.checkNotNull(handle, "handle");
    }

    @Override
    public int guess() {
        return overrideGuess ? guess0() : delegate.guess();
    }

    @Override
    public void reset(ChannelConfig config) {
        overrideGuess = ((KQueueChannelConfig) config).getRcvAllocTransportProvidesGuess();
        delegate.reset(config);
    }

    @Override
    public void incMessagesRead(int numMessages) {
        delegate.incMessagesRead(numMessages);
    }

    @Override
    public ByteBuf allocate(ByteBufAllocator alloc) {
        // We need to ensure we always allocate a direct ByteBuf as we can only use a direct buffer to read via JNI.
        preferredDirectByteBufAllocator.updateAllocator(alloc);
        return overrideGuess ? preferredDirectByteBufAllocator.alluxio.shaded.client.io.uffer(guess0()) :
                delegate.allocate(preferredDirectByteBufAllocator);
    }

    @Override
    public void lastBytesRead(int bytes) {
        numberBytesPending = bytes < 0 ? 0 : max(0, numberBytesPending - bytes);
        delegate.lastBytesRead(bytes);
    }

    @Override
    public int lastBytesRead() {
        return delegate.lastBytesRead();
    }

    @Override
    public void attemptedBytesRead(int bytes) {
        delegate.attemptedBytesRead(bytes);
    }

    @Override
    public int attemptedBytesRead() {
        return delegate.attemptedBytesRead();
    }

    @Override
    public void readComplete() {
        delegate.readComplete();
    }

    @Override
    public boolean continueReading(UncheckedBooleanSupplier maybeMoreDataSupplier) {
        return delegate.continueReading(maybeMoreDataSupplier);
    }

    @Override
    public boolean continueReading() {
        // We must override the supplier which determines if there maybe more data to read.
        return delegate.continueReading(defaultMaybeMoreDataSupplier);
    }

    void readEOF() {
        readEOF = true;
    }

    boolean isReadEOF() {
        return readEOF;
    }

    void numberBytesPending(long numberBytesPending) {
        this.numberBytesPending = numberBytesPending;
    }

    boolean maybeMoreDataToRead() {
        /**
         * kqueue with EV_CLEAR flag set requires that we read until we consume "data" bytes
         * (see <a href="https://www.freebsd.alluxio.shaded.client.org.cgi/man.cgi?kqueue">kqueue man</a>). However in order to
         * respect auto read we supporting reading to stop if auto read is off. If auto read is on we force reading to
         * continue to avoid a {@link StackOverflowError} between channelReadComplete and reading from the
         * channel. It is expected that the {@link #KQueueSocketChannel} implementations will track if all data was not
         * read, and will force a EVFILT_READ ready event.
         *
         * It is assumed EOF is handled externally by checking {@link #isReadEOF()}.
         */
        return numberBytesPending != 0;
    }

    private int guess0() {
        return (int) min(numberBytesPending, Integer.MAX_VALUE);
    }
}
