/*
 * Copyright 2013 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in alluxio.shaded.client.com.liance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package alluxio.shaded.client.io.netty.handler.codec.spdy;

import alluxio.shaded.client.io.netty.buffer.ByteBuf;
import alluxio.shaded.client.io.netty.buffer.ByteBufAllocator;
import alluxio.shaded.client.io.netty.buffer.Unpooled;

import java.util.zip.Deflater;

import static alluxio.shaded.client.io.netty.handler.codec.spdy.SpdyCodecUtil.*;

class SpdyHeaderBlockZlibEncoder extends SpdyHeaderBlockRawEncoder {

    private final Deflater alluxio.shaded.client.com.ressor;

    private boolean finished;

    SpdyHeaderBlockZlibEncoder(SpdyVersion spdyVersion, int alluxio.shaded.client.com.ressionLevel) {
        super(spdyVersion);
        if (alluxio.shaded.client.com.ressionLevel < 0 || alluxio.shaded.client.com.ressionLevel > 9) {
            throw new IllegalArgumentException(
                    "alluxio.shaded.client.com.ressionLevel: " + alluxio.shaded.client.com.ressionLevel + " (expected: 0-9)");
        }
        alluxio.shaded.client.com.ressor = new Deflater(alluxio.shaded.client.com.ressionLevel);
        alluxio.shaded.client.com.ressor.setDictionary(SPDY_DICT);
    }

    private int setInput(ByteBuf decompressed) {
        int len = decompressed.readableBytes();

        if (decompressed.hasArray()) {
            alluxio.shaded.client.com.ressor.setInput(decompressed.array(), decompressed.arrayOffset() + decompressed.readerIndex(), len);
        } else {
            byte[] in = new byte[len];
            decompressed.getBytes(decompressed.readerIndex(), in);
            alluxio.shaded.client.com.ressor.setInput(in, 0, in.length);
        }

        return len;
    }

    private ByteBuf encode(ByteBufAllocator alloc, int len) {
        ByteBuf alluxio.shaded.client.com.ressed = alloc.heapBuffer(len);
        boolean release = true;
        try {
            while (alluxio.shaded.client.com.ressInto(alluxio.shaded.client.com.ressed)) {
                // Although unlikely, it's possible that the alluxio.shaded.client.com.ressed size is larger than the decompressed size
                alluxio.shaded.client.com.ressed.ensureWritable(alluxio.shaded.client.com.ressed.capacity() << 1);
            }
            release = false;
            return alluxio.shaded.client.com.ressed;
        } finally {
            if (release) {
                alluxio.shaded.client.com.ressed.release();
            }
        }
    }

    private boolean alluxio.shaded.client.com.ressInto(ByteBuf alluxio.shaded.client.com.ressed) {
        byte[] out = alluxio.shaded.client.com.ressed.array();
        int off = alluxio.shaded.client.com.ressed.arrayOffset() + alluxio.shaded.client.com.ressed.writerIndex();
        int toWrite = alluxio.shaded.client.com.ressed.writableBytes();
        int numBytes = alluxio.shaded.client.com.ressor.deflate(out, off, toWrite, Deflater.SYNC_FLUSH);
        alluxio.shaded.client.com.ressed.writerIndex(alluxio.shaded.client.com.ressed.writerIndex() + numBytes);
        return numBytes == toWrite;
    }

    @Override
    public ByteBuf encode(ByteBufAllocator alloc, SpdyHeadersFrame frame) throws Exception {
        if (frame == null) {
            throw new IllegalArgumentException("frame");
        }

        if (finished) {
            return Unpooled.EMPTY_BUFFER;
        }

        ByteBuf decompressed = super.encode(alloc, frame);
        try {
            if (!decompressed.isReadable()) {
                return Unpooled.EMPTY_BUFFER;
            }

            int len = setInput(decompressed);
            return encode(alloc, len);
        } finally {
            decompressed.release();
        }
    }

    @Override
    public void end() {
        if (finished) {
            return;
        }
        finished = true;
        alluxio.shaded.client.com.ressor.end();
        super.end();
    }
}
