/*
 * Copyright 2018, OpenCensus Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.io.opencensus.trace.internal;

import alluxio.shaded.client.io.opencensus.alluxio.shaded.client.com.on.Internal;
import alluxio.shaded.client.io.opencensus.internal.Utils;

/**
 * Helper class to convert/cast between for {@link alluxio.shaded.client.io.opencensus.trace.MessageEvent} and {@link
 * alluxio.shaded.client.io.opencensus.trace.NetworkEvent}.
 */
@Internal
@SuppressWarnings("deprecation")
public final class BaseMessageEventUtils {
  /**
   * Cast or convert a {@link alluxio.shaded.client.io.opencensus.trace.BaseMessageEvent} to {@link
   * alluxio.shaded.client.io.opencensus.trace.MessageEvent}.
   *
   * <p>Warning: if the input is a {@code alluxio.shaded.client.io.opencensus.trace.NetworkEvent} and contains {@code
   * kernelTimestamp} information, this information will be dropped.
   *
   * @param event the {@code BaseMessageEvent} that is being cast or converted.
   * @return a {@code MessageEvent} representation of the input.
   */
  public static alluxio.shaded.client.io.opencensus.trace.MessageEvent asMessageEvent(
      alluxio.shaded.client.io.opencensus.trace.BaseMessageEvent event) {
    Utils.checkNotNull(event, "event");
    if (event instanceof alluxio.shaded.client.io.opencensus.trace.MessageEvent) {
      return (alluxio.shaded.client.io.opencensus.trace.MessageEvent) event;
    }
    alluxio.shaded.client.io.opencensus.trace.NetworkEvent networkEvent = (alluxio.shaded.client.io.opencensus.trace.NetworkEvent) event;
    alluxio.shaded.client.io.opencensus.trace.MessageEvent.Type type =
        (networkEvent.getType() == alluxio.shaded.client.io.opencensus.trace.NetworkEvent.Type.RECV)
            ? alluxio.shaded.client.io.opencensus.trace.MessageEvent.Type.RECEIVED
            : alluxio.shaded.client.io.opencensus.trace.MessageEvent.Type.SENT;
    return alluxio.shaded.client.io.opencensus.trace.MessageEvent.builder(type, networkEvent.getMessageId())
        .setUncompressedMessageSize(networkEvent.getUncompressedMessageSize())
        .setCompressedMessageSize(networkEvent.getCompressedMessageSize())
        .build();
  }

  /**
   * Cast or convert a {@link alluxio.shaded.client.io.opencensus.trace.BaseMessageEvent} to {@link
   * alluxio.shaded.client.io.opencensus.trace.NetworkEvent}.
   *
   * @param event the {@code BaseMessageEvent} that is being cast or converted.
   * @return a {@code alluxio.shaded.client.io.opencensus.trace.NetworkEvent} representation of the input.
   */
  public static alluxio.shaded.client.io.opencensus.trace.NetworkEvent asNetworkEvent(
      alluxio.shaded.client.io.opencensus.trace.BaseMessageEvent event) {
    Utils.checkNotNull(event, "event");
    if (event instanceof alluxio.shaded.client.io.opencensus.trace.NetworkEvent) {
      return (alluxio.shaded.client.io.opencensus.trace.NetworkEvent) event;
    }
    alluxio.shaded.client.io.opencensus.trace.MessageEvent messageEvent = (alluxio.shaded.client.io.opencensus.trace.MessageEvent) event;
    alluxio.shaded.client.io.opencensus.trace.NetworkEvent.Type type =
        (messageEvent.getType() == alluxio.shaded.client.io.opencensus.trace.MessageEvent.Type.RECEIVED)
            ? alluxio.shaded.client.io.opencensus.trace.NetworkEvent.Type.RECV
            : alluxio.shaded.client.io.opencensus.trace.NetworkEvent.Type.SENT;
    return alluxio.shaded.client.io.opencensus.trace.NetworkEvent.builder(type, messageEvent.getMessageId())
        .setUncompressedMessageSize(messageEvent.getUncompressedMessageSize())
        .setCompressedMessageSize(messageEvent.getCompressedMessageSize())
        .build();
  }

  private BaseMessageEventUtils() {}
}
