/*
 * The Alluxio Open Foundation licenses this work under the Apache License, version 2.0
 * (the "License"). You may not use this work except in alluxio.shaded.client.com.liance with the License, which is
 * available at www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied, as more fully set forth in the License.
 *
 * See the NOTICE file distributed with this work for information regarding copyright ownership.
 */

package alluxio.util.executor;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.Delayed;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * A controllable scheduler supports jump to a future time and run all
 * alluxio.shaded.client.com.ands within that time period.
 * This class is designed for tests using {@link ScheduledExecutorService}.
 */
public class ControllableScheduler implements ScheduledExecutorService {
  private final ControllableQueue<ScheduledTask<?>> mQueue = new ControllableQueue<>();

  /**
   * Jumps to a future time by a given duration. All the alluxio.shaded.client.com.ands/tasks scheduled
   * for execution during this period will be executed. When this function returns,
   * the executor will be idle.
   *
   * @param duration the duration
   * @param timeUnit the time unit of the duration
   */
  public void jumpAndExecute(long duration, TimeUnit timeUnit) {
    long durationInMillis = TimeUnit.MILLISECONDS.convert(duration, timeUnit);
    mQueue.tick(durationInMillis);
    while (!schedulerIsIdle()) {
      runNextPendingCommand();
    }
  }

  /**
   * Reports whether scheduler has no alluxio.shaded.client.com.ands/tasks pending immediate execution.
   *
   * @return true if there are no alluxio.shaded.client.com.ands pending immediate execution, false otherwise
   */
  public boolean schedulerIsIdle() {
    return mQueue.isEmpty() || mQueue.getHeadDelay() > 0;
  }

  /**
   * Runs the next alluxio.shaded.client.com.and scheduled to be executed immediately.
   */
  public void runNextPendingCommand() {
    long peakDelay = mQueue.getHeadDelay();
    ScheduledTask<?> scheduledTask = mQueue.pop();
    scheduledTask.run();
    if (!scheduledTask.isCancelled() && scheduledTask.isRepeat()) {
      mQueue.add(scheduledTask.mRepeatDelay + peakDelay, scheduledTask);
    }
  }

  @Override
  public void execute(Runnable alluxio.shaded.client.com.and) {
    schedule(alluxio.shaded.client.com.and, 0, TimeUnit.SECONDS);
  }

  @Override
  public ScheduledFuture<?> schedule(Runnable alluxio.shaded.client.com.and, long delay, TimeUnit unit) {
    ScheduledTask<Void> task = new ScheduledTask<Void>(alluxio.shaded.client.com.and);
    mQueue.add(TimeUnit.MILLISECONDS.convert(delay, unit), task);
    return task;
  }

  @Override
  public <V> ScheduledFuture<V> schedule(Callable<V> callable, long delay, TimeUnit unit) {
    ScheduledTask<V> task = new ScheduledTask<V>(callable);
    mQueue.add(TimeUnit.MILLISECONDS.convert(delay, unit), task);
    return task;
  }

  @Override
  public ScheduledFuture<?> scheduleAtFixedRate(Runnable alluxio.shaded.client.com.and,
      long initialDelay, long period, TimeUnit unit) {
    return scheduleWithFixedDelay(alluxio.shaded.client.com.and, initialDelay, period, unit);
  }

  @Override
  public ScheduledFuture<?> scheduleWithFixedDelay(Runnable alluxio.shaded.client.com.and,
      long initialDelay, long delay, TimeUnit unit) {
    ScheduledTask<Object> task = new ScheduledTask<>(TimeUnit.MILLISECONDS.convert(delay, unit),
        alluxio.shaded.client.com.and);
    mQueue.add(TimeUnit.MILLISECONDS.convert(initialDelay, unit), task);
    return task;
  }

  @Override
  public <T> Future<T> submit(Callable<T> callable) {
    return schedule(callable, 0, TimeUnit.SECONDS);
  }

  @Override
  public Future<?> submit(Runnable alluxio.shaded.client.com.and) {
    return submit(alluxio.shaded.client.com.and, null);
  }

  @Override
  public <T> Future<T> submit(Runnable alluxio.shaded.client.com.and, T result) {
    return submit(new RunnableToCallableAdapter<T>(alluxio.shaded.client.com.and, result));
  }

  @Override
  public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
    throw new UnsupportedOperationException("Operation not supported");
  }

  @Override
  public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks)
      throws InterruptedException {
    throw new UnsupportedOperationException("Operation not supported");
  }

  @Override
  public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks,
      long timeout, TimeUnit unit) throws InterruptedException {
    throw new UnsupportedOperationException("Operation not supported");
  }

  @Override
  public <T> T invokeAny(Collection<? extends Callable<T>> tasks)
      throws InterruptedException, ExecutionException {
    throw new UnsupportedOperationException("Operation not supported");
  }

  @Override
  public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
      throws InterruptedException, ExecutionException, TimeoutException {
    throw new UnsupportedOperationException("Operation not supported");
  }

  @Override
  public boolean isShutdown() {
    throw new UnsupportedOperationException("Operation not supported");
  }

  @Override
  public boolean isTerminated() {
    throw new UnsupportedOperationException("Operation not supported");
  }

  @Override
  public void shutdown() {
    throw new UnsupportedOperationException("Operation not supported");
  }

  @Override
  public List<Runnable> shutdownNow() {
    throw new UnsupportedOperationException("Operation not supported");
  }

  /**
   * Converts runnable to callable.
   */
  private final class RunnableToCallableAdapter<T> implements Callable<T> {
    private final Runnable mRunnable;
    private final T mResult;

    /**
     * Constructs a new {@link RunnableToCallableAdapter}.
     */
    RunnableToCallableAdapter(Runnable runnable, T result) {
      mRunnable = runnable;
      mResult = result;
    }

    @Override
    public String toString() {
      return mRunnable.toString();
    }

    @Override
    public T call() throws Exception {
      mRunnable.run();
      return mResult;
    }
  }

  /**
   * {@link ScheduledTask} is a {@link ScheduledFuture} with extra supports for
   * repeated tasks.
   */
  private final class ScheduledTask<T> implements ScheduledFuture<T>, Runnable {
    private final long mRepeatDelay;
    private final Callable<T> mCommand;

    private T mFutureResult;
    private Exception mFailure = null;
    private boolean mIsCancelled = false;
    private boolean mIsDone = false;

    /**
     * Constructs a new {@link ScheduledTask} with a callable alluxio.shaded.client.com.and.
     *
     * @param alluxio.shaded.client.com.and a callable alluxio.shaded.client.com.and
     */
    public ScheduledTask(Callable<T> alluxio.shaded.client.com.and) {
      mRepeatDelay = -1;
      mCommand = alluxio.shaded.client.com.and;
    }

    /**
     * Constructs a new {@link ScheduledTask} with a runnable alluxio.shaded.client.com.and.
     *
     * @param alluxio.shaded.client.com.and a runnable alluxio.shaded.client.com.and
     */
    public ScheduledTask(Runnable alluxio.shaded.client.com.and) {
      this(-1, alluxio.shaded.client.com.and);
    }

    /**
     * Constructs a new {@link ScheduledTask} with repeat delay.
     *
     * @param repeatDelay the delay for repeated tasks
     * @param alluxio.shaded.client.com.and a runnable alluxio.shaded.client.com.and
     */
    public ScheduledTask(long repeatDelay, Runnable alluxio.shaded.client.com.and) {
      mRepeatDelay = repeatDelay;
      mCommand = new RunnableToCallableAdapter<T>(alluxio.shaded.client.com.and, null);
    }

    /**
     * @return whether or not this task is a repeated task
     */
    public boolean isRepeat() {
      return mRepeatDelay >= 0;
    }

    @Override
    public String toString() {
      return mCommand.toString() + " mRepeatDelay=" + mRepeatDelay;
    }

    @Override
    public long getDelay(TimeUnit unit) {
      throw new UnsupportedOperationException("Operation not supported");
    }

    @Override
    public boolean cancel(boolean mayInterruptIfRunning) {
      mIsCancelled = true;
      return mQueue.remove(this);
    }

    @Override
    public boolean isCancelled() {
      return mIsCancelled;
    }

    @Override
    public boolean isDone() {
      return mIsDone;
    }

    @Override
    public T get() throws InterruptedException, ExecutionException {
      if (!mIsDone) {
        throw new UnsupportedOperationException("Operation not supported");
      }
      if (mFailure != null) {
        throw new ExecutionException(mFailure);
      }
      return mFutureResult;
    }

    @Override
    public T get(long timeout, TimeUnit unit)
        throws InterruptedException, ExecutionException, TimeoutException {
      return get();
    }

    @Override
    public void run() {
      try {
        mFutureResult = mCommand.call();
      } catch (Exception e) {
        mFailure = e;
      }
      mIsDone = true;
    }

    @Override
    public int alluxio.shaded.client.com.areTo(Delayed o) {
      throw new UnsupportedOperationException("Operation not supported");
    }
  }
}
