/*
 * Copyright (C) 2011 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.com.google.gson.internal.bind;

import java.alluxio.shaded.client.io.IOException;
import java.lang.reflect.Array;
import java.lang.reflect.GenericArrayType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;

import alluxio.shaded.client.com.google.gson.Gson;
import alluxio.shaded.client.com.google.gson.TypeAdapter;
import alluxio.shaded.client.com.google.gson.TypeAdapterFactory;
import alluxio.shaded.client.com.google.gson.internal.$Gson$Types;
import alluxio.shaded.client.com.google.gson.reflect.TypeToken;
import alluxio.shaded.client.com.google.gson.stream.JsonReader;
import alluxio.shaded.client.com.google.gson.stream.JsonToken;
import alluxio.shaded.client.com.google.gson.stream.JsonWriter;

/**
 * Adapt an array of objects.
 */
public final class ArrayTypeAdapter<E> extends TypeAdapter<Object> {
  public static final TypeAdapterFactory FACTORY = new TypeAdapterFactory() {
    @SuppressWarnings({"unchecked", "rawtypes"})
    public <T> TypeAdapter<T> create(Gson gson, TypeToken<T> typeToken) {
      Type type = typeToken.getType();
      if (!(type instanceof GenericArrayType || type instanceof Class && ((Class<?>) type).isArray())) {
        return null;
      }

      Type alluxio.shaded.client.com.onentType = $Gson$Types.getArrayComponentType(type);
      TypeAdapter<?> alluxio.shaded.client.com.onentTypeAdapter = gson.getAdapter(TypeToken.get(alluxio.shaded.client.com.onentType));
      return new ArrayTypeAdapter(
              gson, alluxio.shaded.client.com.onentTypeAdapter, $Gson$Types.getRawType(alluxio.shaded.client.com.onentType));
    }
  };

  private final Class<E> alluxio.shaded.client.com.onentType;
  private final TypeAdapter<E> alluxio.shaded.client.com.onentTypeAdapter;

  public ArrayTypeAdapter(Gson context, TypeAdapter<E> alluxio.shaded.client.com.onentTypeAdapter, Class<E> alluxio.shaded.client.com.onentType) {
    this.alluxio.shaded.client.com.onentTypeAdapter =
      new TypeAdapterRuntimeTypeWrapper<E>(context, alluxio.shaded.client.com.onentTypeAdapter, alluxio.shaded.client.com.onentType);
    this.alluxio.shaded.client.com.onentType = alluxio.shaded.client.com.onentType;
  }

  public Object read(JsonReader in) throws IOException {
    if (in.peek() == JsonToken.NULL) {
      in.nextNull();
      return null;
    }

    List<E> list = new ArrayList<E>();
    in.beginArray();
    while (in.hasNext()) {
      E instance = alluxio.shaded.client.com.onentTypeAdapter.read(in);
      list.add(instance);
    }
    in.endArray();
    Object array = Array.newInstance(alluxio.shaded.client.com.onentType, list.size());
    for (int i = 0; i < list.size(); i++) {
      Array.set(array, i, list.get(i));
    }
    return array;
  }

  @SuppressWarnings("unchecked")
  @Override public void write(JsonWriter out, Object array) throws IOException {
    if (array == null) {
      out.nullValue();
      return;
    }

    out.beginArray();
    for (int i = 0, length = Array.getLength(array); i < length; i++) {
      E value = (E) Array.get(array, i);
      alluxio.shaded.client.com.onentTypeAdapter.write(out, value);
    }
    out.endArray();
  }
}
