/*
 * The Alluxio Open Foundation licenses this work under the Apache License, version 2.0
 * (the "License"). You may not use this work except in alluxio.shaded.client.com.liance with the License, which is
 * available at www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied, as more fully set forth in the License.
 *
 * See the NOTICE file distributed with this work for information regarding copyright ownership.
 */

package alluxio.security.login;

import alluxio.util.OSUtils;

import alluxio.shaded.client.javax.annotation.concurrent.ThreadSafe;

/**
 * This class provides constants used in JAAS login.
 */
@ThreadSafe
public final class LoginModuleConfigurationUtils {
  /** Login module according to different OS type. */
  public static final String OS_LOGIN_MODULE_NAME;
  /** Class name of Principal according to different OS type. */
  public static final String OS_PRINCIPAL_CLASS_NAME;

  private LoginModuleConfigurationUtils() {} // prevent instantiation

  static {
    OS_LOGIN_MODULE_NAME = getOSLoginModuleName();
    OS_PRINCIPAL_CLASS_NAME = getOSPrincipalClassName();
  }

  /**
   * @return the OS login module class name
   */
  private static String getOSLoginModuleName() {
    if (OSUtils.IBM_JAVA) {
      if (OSUtils.isWindows()) {
        return OSUtils.is64Bit() ? "alluxio.shaded.client.com.ibm.security.auth.module.Win64LoginModule"
            : "alluxio.shaded.client.com.ibm.security.auth.module.NTLoginModule";
      } else if (OSUtils.isAIX()) {
        return OSUtils.is64Bit() ? "alluxio.shaded.client.com.ibm.security.auth.module.AIX64LoginModule"
            : "alluxio.shaded.client.com.ibm.security.auth.module.AIXLoginModule";
      } else {
        return "alluxio.shaded.client.com.ibm.security.auth.module.LinuxLoginModule";
      }
    } else {
      return OSUtils.isWindows() ? "alluxio.shaded.client.com.sun.security.auth.module.NTLoginModule"
          : "alluxio.shaded.client.com.sun.security.auth.module.UnixLoginModule";
    }
  }

  /**
   * @return the OS principal class name
   */
  private static String getOSPrincipalClassName() {
    String principalClassName;
    if (OSUtils.IBM_JAVA) {
      if (OSUtils.is64Bit()) {
        principalClassName = "alluxio.shaded.client.com.ibm.security.auth.UsernamePrincipal";
      } else {
        if (OSUtils.isWindows()) {
          principalClassName = "alluxio.shaded.client.com.ibm.security.auth.NTUserPrincipal";
        } else if (OSUtils.isAIX()) {
          principalClassName = "alluxio.shaded.client.com.ibm.security.auth.AIXPrincipal";
        } else {
          principalClassName = "alluxio.shaded.client.com.ibm.security.auth.LinuxPrincipal";
        }
      }
    } else {
      principalClassName = OSUtils.isWindows() ? "alluxio.shaded.client.com.sun.security.auth.NTUserPrincipal"
          : "alluxio.shaded.client.com.sun.security.auth.UnixPrincipal";
    }
    return principalClassName;
  }
}
