/*
 * Copyright 2013 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in alluxio.shaded.client.com.liance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package alluxio.shaded.client.io.netty.example.http.helloworld;

import alluxio.shaded.client.io.netty.buffer.Unpooled;
import alluxio.shaded.client.io.netty.channel.ChannelFuture;
import alluxio.shaded.client.io.netty.channel.ChannelFutureListener;
import alluxio.shaded.client.io.netty.channel.ChannelHandlerContext;
import alluxio.shaded.client.io.netty.channel.SimpleChannelInboundHandler;
import alluxio.shaded.client.io.netty.handler.codec.http.DefaultFullHttpResponse;
import alluxio.shaded.client.io.netty.handler.codec.http.FullHttpResponse;
import alluxio.shaded.client.io.netty.handler.codec.http.HttpObject;
import alluxio.shaded.client.io.netty.handler.codec.http.HttpRequest;
import alluxio.shaded.client.io.netty.handler.codec.http.HttpUtil;

import static alluxio.shaded.client.io.netty.handler.codec.http.HttpHeaderNames.CONNECTION;
import static alluxio.shaded.client.io.netty.handler.codec.http.HttpHeaderNames.CONTENT_LENGTH;
import static alluxio.shaded.client.io.netty.handler.codec.http.HttpHeaderNames.CONTENT_TYPE;
import static alluxio.shaded.client.io.netty.handler.codec.http.HttpHeaderValues.CLOSE;
import static alluxio.shaded.client.io.netty.handler.codec.http.HttpHeaderValues.KEEP_ALIVE;
import static alluxio.shaded.client.io.netty.handler.codec.http.HttpHeaderValues.TEXT_PLAIN;
import static alluxio.shaded.client.io.netty.handler.codec.http.HttpResponseStatus.OK;

public class HttpHelloWorldServerHandler extends SimpleChannelInboundHandler<HttpObject> {
    private static final byte[] CONTENT = { 'H', 'e', 'l', 'l', 'o', ' ', 'W', 'o', 'r', 'l', 'd' };

    @Override
    public void channelReadComplete(ChannelHandlerContext ctx) {
        ctx.flush();
    }

    @Override
    public void channelRead0(ChannelHandlerContext ctx, HttpObject msg) {
        if (msg instanceof HttpRequest) {
            HttpRequest req = (HttpRequest) msg;

            boolean keepAlive = HttpUtil.isKeepAlive(req);
            FullHttpResponse response = new DefaultFullHttpResponse(req.protocolVersion(), OK,
                                                                    Unpooled.wrappedBuffer(CONTENT));
            response.headers()
                    .set(CONTENT_TYPE, TEXT_PLAIN)
                    .setInt(CONTENT_LENGTH, response.content().readableBytes());

            if (keepAlive) {
                if (!req.protocolVersion().isKeepAliveDefault()) {
                    response.headers().set(CONNECTION, KEEP_ALIVE);
                }
            } else {
                // Tell the client we're going to close the connection.
                response.headers().set(CONNECTION, CLOSE);
            }

            ChannelFuture f = ctx.write(response);

            if (!keepAlive) {
                f.addListener(ChannelFutureListener.CLOSE);
            }
        }
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        cause.printStackTrace();
        ctx.close();
    }
}
