/*
 * Decompiled with CFR 0.152.
 */
package alluxio;

import alluxio.Client;
import alluxio.ClientContext;
import alluxio.annotation.SuppressFBWarnings;
import alluxio.conf.AlluxioConfiguration;
import alluxio.conf.PropertyKey;
import alluxio.exception.ExceptionMessage;
import alluxio.exception.ServiceNotFoundException;
import alluxio.exception.runtime.AlluxioRuntimeException;
import alluxio.exception.runtime.FailedPreconditionRuntimeException;
import alluxio.exception.status.AlluxioStatusException;
import alluxio.exception.status.FailedPreconditionException;
import alluxio.exception.status.NotFoundException;
import alluxio.exception.status.UnauthenticatedException;
import alluxio.exception.status.UnavailableException;
import alluxio.grpc.GetServiceVersionPRequest;
import alluxio.grpc.GrpcChannel;
import alluxio.grpc.GrpcChannelBuilder;
import alluxio.grpc.GrpcServerAddress;
import alluxio.grpc.ServiceType;
import alluxio.grpc.ServiceVersionClientServiceGrpc;
import alluxio.metrics.Metric;
import alluxio.metrics.MetricsSystem;
import alluxio.retry.RetryPolicy;
import alluxio.retry.RetryUtils;
import alluxio.shaded.client.com.codahale.metrics.Timer;
import alluxio.shaded.client.com.google.common.base.Preconditions;
import alluxio.shaded.client.io.grpc.Status;
import alluxio.shaded.client.io.grpc.StatusRuntimeException;
import alluxio.shaded.client.javax.annotation.concurrent.ThreadSafe;
import alluxio.util.CommonUtils;
import alluxio.util.SecurityUtils;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.nio.channels.UnresolvedAddressException;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public abstract class AbstractClient
implements Client {
    private static final Logger LOG = LoggerFactory.getLogger(AbstractClient.class);
    private final Supplier<RetryPolicy> mRetryPolicySupplier;
    protected GrpcServerAddress mServerAddress = null;
    protected GrpcChannel mChannel;
    @SuppressFBWarnings(value={"IS2_INCONSISTENT_SYNC"}, justification="the error seems a bug in findbugs")
    protected ServiceVersionClientServiceGrpc.ServiceVersionClientServiceBlockingStub mVersionService;
    protected boolean mConnected = false;
    protected volatile boolean mClosed = false;
    protected long mServiceVersion;
    protected ClientContext mContext;
    private final long mRpcThreshold;

    protected AbstractClient(ClientContext context) {
        this(context, RetryUtils::defaultClientRetry);
    }

    protected AbstractClient(ClientContext context, Supplier<RetryPolicy> retryPolicySupplier) {
        this.mContext = Preconditions.checkNotNull(context, "context");
        this.mRetryPolicySupplier = retryPolicySupplier;
        this.mServiceVersion = -1L;
        this.mRpcThreshold = this.mContext.getClusterConf().getMs(PropertyKey.USER_LOGGING_THRESHOLD);
    }

    protected abstract ServiceType getRemoteServiceType();

    protected long getRemoteServiceVersion() throws AlluxioStatusException {
        try {
            return this.mVersionService.getServiceVersion(GetServiceVersionPRequest.newBuilder().setServiceType(this.getRemoteServiceType()).setAllowedOnStandbyMasters(true).build()).getVersion();
        }
        catch (Throwable t) {
            throw AlluxioStatusException.fromThrowable(t);
        }
    }

    protected abstract String getServiceName();

    protected abstract long getServiceVersion();

    protected void checkVersion(long clientVersion) throws IOException {
        if (this.mServiceVersion == -1L) {
            this.mServiceVersion = this.getRemoteServiceVersion();
            if (this.mServiceVersion != clientVersion) {
                throw new IOException(ExceptionMessage.INCOMPATIBLE_VERSION.getMessage(this.getServiceName(), clientVersion, this.mServiceVersion));
            }
        }
    }

    protected void afterConnect() throws IOException {
    }

    protected void beforeConnect() throws IOException {
        if (!this.isConnected()) {
            this.mContext.loadConfIfNotLoaded(this.getConfAddress());
        }
    }

    protected void afterDisconnect() {
    }

    protected void beforeDisconnect() {
    }

    public synchronized void connectWithRuntimeException() {
        if (this.mClosed) {
            throw new FailedPreconditionRuntimeException("Failed to connect: client has been closed");
        }
        try {
            this.connect();
        }
        catch (AlluxioStatusException e) {
            throw AlluxioRuntimeException.from(e);
        }
    }

    @Override
    public synchronized void connect() throws AlluxioStatusException {
        if (this.mConnected) {
            return;
        }
        this.disconnect();
        Preconditions.checkState(!this.mClosed, "Client is closed, will not try to connect.");
        Throwable lastConnectFailure = null;
        RetryPolicy retryPolicy = this.mRetryPolicySupplier.get();
        while (retryPolicy.attempt()) {
            if (this.mClosed) {
                throw new FailedPreconditionException("Failed to connect: client has been closed");
            }
            try {
                this.mServerAddress = this.queryGrpcServerAddress();
            }
            catch (UnavailableException e) {
                LOG.debug("Failed to determine {} rpc address ({}): {}", new Object[]{this.getServiceName(), retryPolicy.getAttemptCount(), e.toString()});
                continue;
            }
            try {
                this.beforeConnect();
                LOG.debug("Alluxio client (version {}) is trying to connect with {} @ {}", new Object[]{"2.9.5", this.getServiceName(), this.mServerAddress});
                AlluxioConfiguration conf = this.mContext.getClusterConf();
                this.mChannel = GrpcChannelBuilder.newBuilder(this.mServerAddress, conf).setSubject(this.mContext.getSubject()).build();
                this.mVersionService = ServiceVersionClientServiceGrpc.newBlockingStub(this.mChannel);
                this.mConnected = true;
                this.afterConnect();
                this.checkVersion(this.getServiceVersion());
                LOG.debug("Alluxio client (version {}) is connected with {} @ {}", new Object[]{"2.9.5", this.getServiceName(), this.mServerAddress});
                return;
            }
            catch (IOException e) {
                LOG.debug("Failed to connect ({}) with {} @ {}", new Object[]{retryPolicy.getAttemptCount(), this.getServiceName(), this.mServerAddress, e});
                lastConnectFailure = e;
                if (e instanceof UnauthenticatedException) {
                    this.mContext.getUserState().relogin();
                }
                if (!(e instanceof NotFoundException)) continue;
                break;
            }
        }
        if (this.mChannel != null) {
            this.mChannel.shutdown();
        }
        if (this.mServerAddress == null) {
            throw new UnavailableException(String.format("Failed to determine address for %s after %s attempts", this.getServiceName(), retryPolicy.getAttemptCount()));
        }
        if (lastConnectFailure instanceof UnauthenticatedException) {
            throw (AlluxioStatusException)lastConnectFailure;
        }
        if (lastConnectFailure instanceof NotFoundException) {
            throw new NotFoundException(lastConnectFailure.getMessage(), new ServiceNotFoundException(lastConnectFailure.getMessage(), lastConnectFailure));
        }
        throw new UnavailableException(String.format("Failed to connect to master (%s) after %s attempts.Please check if Alluxio master is currently running on \"%s\". Service=\"%s\"", this.mServerAddress, retryPolicy.getAttemptCount(), this.mServerAddress, this.getServiceName()), lastConnectFailure);
    }

    @Override
    public synchronized void disconnect() {
        if (this.mConnected) {
            Preconditions.checkNotNull(this.mChannel, "The client channel should never be null when the client is connected");
            LOG.debug("Disconnecting from the {} @ {}", (Object)this.getServiceName(), (Object)this.mServerAddress);
            this.beforeDisconnect();
            this.mChannel.shutdown();
            this.mConnected = false;
            this.afterDisconnect();
        }
    }

    @Override
    public synchronized boolean isConnected() {
        return this.mConnected;
    }

    @Override
    public synchronized void close() {
        this.disconnect();
        this.mClosed = true;
    }

    protected abstract GrpcServerAddress queryGrpcServerAddress() throws UnavailableException;

    @Override
    public synchronized SocketAddress getRemoteSockAddress() throws UnavailableException {
        if (this.mServerAddress == null) {
            this.mServerAddress = this.queryGrpcServerAddress();
        }
        return this.mServerAddress.getSocketAddress();
    }

    @Override
    public synchronized String getRemoteHostName() throws UnavailableException {
        if (this.mServerAddress == null) {
            this.mServerAddress = this.queryGrpcServerAddress();
        }
        return this.mServerAddress.getHostName();
    }

    @Override
    public synchronized InetSocketAddress getConfAddress() throws UnavailableException {
        SocketAddress sockAddress;
        if (this.mServerAddress == null) {
            this.mServerAddress = this.queryGrpcServerAddress();
        }
        if ((sockAddress = this.mServerAddress.getSocketAddress()) instanceof InetSocketAddress) {
            return (InetSocketAddress)sockAddress;
        }
        throw new UnavailableException("Remote is not an InetSockAddress");
    }

    protected synchronized <V> V retryRPC(RpcCallable<V> rpc, Logger logger, String rpcName, String description, Object ... args) throws AlluxioStatusException {
        return this.retryRPC(this.mRetryPolicySupplier.get(), rpc, logger, rpcName, description, args);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    protected synchronized <V> V retryRPC(RetryPolicy retryPolicy, RpcCallable<V> rpc, Logger logger, String rpcName, String description, Object ... args) throws AlluxioStatusException {
        String debugDesc = logger.isDebugEnabled() ? String.format(description, args) : null;
        long startMs = System.currentTimeMillis();
        logger.debug("Enter: {}({})", (Object)rpcName, (Object)debugDesc);
        try (Timer.Context ctx = MetricsSystem.timer(this.getQualifiedMetricName(rpcName)).time();){
            V ret = this.retryRPCInternal(retryPolicy, rpc, () -> {
                MetricsSystem.counter(this.getQualifiedRetryMetricName(rpcName)).inc();
                return null;
            });
            long duration = System.currentTimeMillis() - startMs;
            logger.debug("Exit (OK): {}({}) in {} ms", new Object[]{rpcName, debugDesc, duration});
            if (duration >= this.mRpcThreshold) {
                logger.warn("{}({}) returned {} in {} ms (>={} ms)", new Object[]{rpcName, String.format(description, args), CommonUtils.summarizeCollection(ret), duration, this.mRpcThreshold});
            }
            V v = ret;
            return v;
        }
        catch (Exception e) {
            long duration = System.currentTimeMillis() - startMs;
            MetricsSystem.counter(this.getQualifiedFailureMetricName(rpcName)).inc();
            logger.debug("Exit (ERROR): {}({}) in {} ms: {}", new Object[]{rpcName, debugDesc, duration, e.toString()});
            if (duration < this.mRpcThreshold) throw e;
            logger.warn("{}({}) exits with exception [{}] in {} ms (>={}ms)", new Object[]{rpcName, String.format(description, args), e, duration, this.mRpcThreshold});
            throw e;
        }
    }

    private synchronized <V> V retryRPCInternal(RetryPolicy retryPolicy, RpcCallable<V> rpc, Supplier<Void> onRetry) throws AlluxioStatusException {
        AlluxioStatusException ex = null;
        while (retryPolicy.attempt()) {
            if (this.mClosed) {
                throw new FailedPreconditionException("Client is closed");
            }
            this.connect();
            try {
                return rpc.call();
            }
            catch (StatusRuntimeException e) {
                AlluxioStatusException se = AlluxioStatusException.fromStatusRuntimeException(e);
                if (se.getStatusCode() != Status.Code.UNAVAILABLE && se.getStatusCode() != Status.Code.CANCELLED && se.getStatusCode() != Status.Code.UNAUTHENTICATED && se.getStatusCode() != Status.Code.UNIMPLEMENTED && !(e.getCause() instanceof UnresolvedAddressException)) {
                    throw se;
                }
                ex = se;
                LOG.debug("Rpc failed ({}): ", (Object)retryPolicy.getAttemptCount(), (Object)ex);
                onRetry.get();
                this.disconnect();
            }
        }
        throw new UnavailableException(String.format("Failed after %d attempts: %s", retryPolicy.getAttemptCount(), ex), ex);
    }

    private String getQualifiedMetricName(String metricName) {
        try {
            if (SecurityUtils.isAuthenticationEnabled(this.mContext.getClusterConf()) && this.mContext.getUserState().getUser() != null) {
                return Metric.getMetricNameWithTags(metricName, "User", this.mContext.getUserState().getUser().getName());
            }
            return metricName;
        }
        catch (IOException e) {
            return metricName;
        }
    }

    private String getQualifiedRetryMetricName(String metricName) {
        return this.getQualifiedMetricName(metricName + "Retries");
    }

    private String getQualifiedFailureMetricName(String metricName) {
        return this.getQualifiedMetricName(metricName + "Failures");
    }

    @Override
    public boolean isClosed() {
        return this.mClosed;
    }

    @FunctionalInterface
    protected static interface RpcCallable<V> {
        public V call() throws StatusRuntimeException;
    }
}

