/*
 * Copyright 2020 The gRPC Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.io.grpc.internal;

import alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.base.MoreObjects;
import alluxio.shaded.client.io.grpc.BinaryLog;
import alluxio.shaded.client.io.grpc.BindableService;
import alluxio.shaded.client.io.grpc.CompressorRegistry;
import alluxio.shaded.client.io.grpc.DecompressorRegistry;
import alluxio.shaded.client.io.grpc.HandlerRegistry;
import alluxio.shaded.client.io.grpc.Server;
import alluxio.shaded.client.io.grpc.ServerBuilder;
import alluxio.shaded.client.io.grpc.ServerInterceptor;
import alluxio.shaded.client.io.grpc.ServerServiceDefinition;
import alluxio.shaded.client.io.grpc.ServerStreamTracer;
import alluxio.shaded.client.io.grpc.ServerTransportFilter;
import java.alluxio.shaded.client.io.File;
import java.alluxio.shaded.client.io.InputStream;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import alluxio.shaded.client.javax.annotation.Nullable;

/**
 * A {@link ServerBuilder} that delegates all its builder method to another builder by default.
 *
 * <p>Temporarily duplicates alluxio.shaded.client.io.grpc.ForwardingServerBuilder (temporarily package-private)
 * to fix ABI backward alluxio.shaded.client.com.atibility.

 * @param <T> The concrete type of this builder.
 * @see <a href="https://github.alluxio.shaded.client.com.grpc/grpc-java/issues/7211">grpc/grpc-java#7211</a>
 */
public abstract class AbstractServerImplBuilder
    <T extends AbstractServerImplBuilder<T>> extends ServerBuilder<T> {

  /** The default constructor. */
  protected AbstractServerImplBuilder() {}

  /**
   * This method serves to force sub classes to "hide" this static factory.
   */
  public static ServerBuilder<?> forPort(int port) {
    throw new UnsupportedOperationException("Subclass failed to hide static factory");
  }

  /**
   * Returns the delegated {@code ServerBuilder}.
   */
  protected abstract ServerBuilder<?> delegate();

  @Override
  public T directExecutor() {
    delegate().directExecutor();
    return thisT();
  }

  @Override
  public T executor(@Nullable Executor executor) {
    delegate().executor(executor);
    return thisT();
  }

  @Override
  public T addService(ServerServiceDefinition service) {
    delegate().addService(service);
    return thisT();
  }

  @Override
  public T addService(BindableService bindableService) {
    delegate().addService(bindableService);
    return thisT();
  }

  @Override
  public T intercept(ServerInterceptor interceptor) {
    delegate().intercept(interceptor);
    return thisT();
  }

  @Override
  public T addTransportFilter(ServerTransportFilter filter) {
    delegate().addTransportFilter(filter);
    return thisT();
  }

  @Override
  public T addStreamTracerFactory(ServerStreamTracer.Factory factory) {
    delegate().addStreamTracerFactory(factory);
    return thisT();
  }

  @Override
  public T fallbackHandlerRegistry(@Nullable HandlerRegistry fallbackRegistry) {
    delegate().fallbackHandlerRegistry(fallbackRegistry);
    return thisT();
  }

  @Override
  public T useTransportSecurity(File certChain, File privateKey) {
    delegate().useTransportSecurity(certChain, privateKey);
    return thisT();
  }

  @Override
  public T useTransportSecurity(InputStream certChain, InputStream privateKey) {
    delegate().useTransportSecurity(certChain, privateKey);
    return thisT();
  }

  @Override
  public T decompressorRegistry(@Nullable DecompressorRegistry registry) {
    delegate().decompressorRegistry(registry);
    return thisT();
  }

  @Override
  public T alluxio.shaded.client.com.ressorRegistry(@Nullable CompressorRegistry registry) {
    delegate().alluxio.shaded.client.com.ressorRegistry(registry);
    return thisT();
  }

  @Override
  public T handshakeTimeout(long timeout, TimeUnit unit) {
    delegate().handshakeTimeout(timeout, unit);
    return thisT();
  }

  @Override
  public T maxInboundMessageSize(int bytes) {
    delegate().maxInboundMessageSize(bytes);
    return thisT();
  }

  @Override
  public T maxInboundMetadataSize(int bytes) {
    delegate().maxInboundMetadataSize(bytes);
    return thisT();
  }

  @Override
  public T setBinaryLog(BinaryLog binaryLog) {
    delegate().setBinaryLog(binaryLog);
    return thisT();
  }

  /**
   * Returns the {@link Server} built by the delegate by default. Overriding method can return
   * different value.
   */
  @Override
  public Server build() {
    return delegate().build();
  }

  @Override
  public String toString() {
    return MoreObjects.toStringHelper(this).add("delegate", delegate()).toString();
  }

  /**
   * Returns the correctly typed version of the builder.
   */
  protected final T thisT() {
    @SuppressWarnings("unchecked")
    T thisT = (T) this;
    return thisT;
  }
}
