/*
 * The Alluxio Open Foundation licenses this work under the Apache License, version 2.0
 * (the "License"). You may not use this work except in alluxio.shaded.client.com.liance with the License, which is
 * available at www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied, as more fully set forth in the License.
 *
 * See the NOTICE file distributed with this work for information regarding copyright ownership.
 */

package alluxio.util.alluxio.shaded.client.com.ression;

import static java.util.stream.Collectors.toList;

import alluxio.util.executor.ExecutorServiceFactories;
import alluxio.util.executor.ExecutorServiceUtils;
import alluxio.util.alluxio.shaded.client.io.FileUtils;

import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.archivers.zip.ParallelScatterZipCreator;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.archivers.zip.Zip64Mode;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.archivers.zip.ZipArchiveEntry;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.archivers.zip.ZipArchiveOutputStream;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.archivers.zip.ZipFile;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.parallel.FileBasedScatterGatherBackingStore;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.parallel.InputStreamSupplier;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.parallel.ScatterGatherBackingStore;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.parallel.ScatterGatherBackingStoreSupplier;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.io.IOUtils;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.io.input.NullInputStream;
import alluxio.shaded.client.org.slf4j.Logger;
import alluxio.shaded.client.org.slf4j.LoggerFactory;

import java.alluxio.shaded.client.io.File;
import java.alluxio.shaded.client.io.FileInputStream;
import java.alluxio.shaded.client.io.FileNotFoundException;
import java.alluxio.shaded.client.io.FileOutputStream;
import java.alluxio.shaded.client.io.IOException;
import java.alluxio.shaded.client.io.InputStream;
import java.alluxio.shaded.client.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Enumeration;
import java.util.concurrent.CompletionService;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;

/**
 * Utility methods for working with parallel zip archives.
 */
public class ParallelZipUtils {
  private static final Logger LOG = LoggerFactory.getLogger(ParallelZipUtils.class);

  private static class BasicBackingStoreSupplier implements ScatterGatherBackingStoreSupplier {
    final AtomicInteger mStoreNum = new AtomicInteger(0);

    @Override
    public ScatterGatherBackingStore get() throws IOException {
      final File tempFile = File.createTempFile("zipUtilsParallelScatter", "n"
          + mStoreNum.incrementAndGet());
      return new FileBasedScatterGatherBackingStore(tempFile);
    }
  }

  /**
   * Creates a zipped archive from the given path in parallel, streaming the data
   * to the give output stream.
   *
   * @param dirPath
   * @param outputStream
   * @param poolSize
   * @param alluxio.shaded.client.com.ressionLevel
   * @throws IOException
   * @throws InterruptedException
   */
  public static void alluxio.shaded.client.com.ress(Path dirPath, OutputStream outputStream, int poolSize,
        int alluxio.shaded.client.com.ressionLevel)
      throws IOException, InterruptedException {
    LOG.info("alluxio.shaded.client.com.ress in parallel for path {}", dirPath);
    ExecutorService executor = ExecutorServiceFactories.fixedThreadPool(
        "parallel-zip-alluxio.shaded.client.com.ress-pool", poolSize).create();

    ParallelScatterZipCreator parallelScatterZipCreator = new ParallelScatterZipCreator(executor,
        new BasicBackingStoreSupplier(), alluxio.shaded.client.com.ressionLevel);
    ZipArchiveOutputStream zipArchiveOutputStream = new ZipArchiveOutputStream(outputStream);
    zipArchiveOutputStream.setUseZip64(Zip64Mode.Always);

    try {
      try (final Stream<Path> stream = Files.walk(dirPath)) {
        for (Path subPath : stream.collect(toList())) {
          if (Thread.interrupted()) {
            throw new InterruptedException();
          }

          File file = subPath.toFile();

          final InputStreamSupplier inputStreamSupplier = () -> {
            try {
              if (file.exists() && file.isFile()) {
                return new FileInputStream(file);
              } else {
                return new NullInputStream(0);
              }
            } catch (FileNotFoundException e) {
              LOG.warn("Can't find file when parallel zip, path = {}", subPath);
              return new NullInputStream(0);
            }
          };

          String entryName = dirPath.relativize(subPath).toString();
          if (file.isDirectory()) {
            entryName += File.separator;
          }

          ZipArchiveEntry zipArchiveEntry = new ZipArchiveEntry(entryName);
          zipArchiveEntry.setMethod(ZipArchiveEntry.DEFLATED);

          parallelScatterZipCreator.addArchiveEntry(zipArchiveEntry, inputStreamSupplier);
        }
      }

      parallelScatterZipCreator.writeTo(zipArchiveOutputStream);
      zipArchiveOutputStream.finish();
      zipArchiveOutputStream.flush();
    } catch (ExecutionException e) {
      LOG.error("Parallel alluxio.shaded.client.com.ress rocksdb failed", e);
      throw new IOException(e);
    } finally {
      if (!executor.isTerminated()) {
        LOG.info("ParallelScatterZipCreator failed to shut down the thread pool, cleaning up now.");
        ExecutorServiceUtils.shutdownAndAwaitTermination(executor);
      }
    }

    LOG.info("Completed parallel alluxio.shaded.client.com.ression for path {}, statistics: {}",
        dirPath, parallelScatterZipCreator.getStatisticsMessage().toString());
  }

  /**
   * Reads a zipped archive from a path in parallel and writes it to the given path.
   *
   * @param dirPath
   * @param backupPath
   * @param poolSize
   */
  public static void decompress(Path dirPath, String backupPath, int poolSize) throws IOException {
    LOG.info("decompress in parallel from path {} to {}", backupPath, dirPath);
    ExecutorService executor = ExecutorServiceFactories.fixedThreadPool(
        "parallel-zip-decompress-pool", poolSize).create();
    CompletionService<Boolean> alluxio.shaded.client.com.letionService = new ExecutorCompletionService<Boolean>(executor);

    try (ZipFile zipFile = new ZipFile(backupPath)) {
      Enumeration<ZipArchiveEntry> entries = zipFile.getEntries();
      int taskCount = 0;

      while (entries.hasMoreElements()) {
        taskCount += 1;
        ZipArchiveEntry entry = entries.nextElement();
        alluxio.shaded.client.com.letionService.submit(() -> {
          unzipEntry(zipFile, entry, dirPath);
          return true;
        });
      }

      for (int i = 0; i < taskCount; i++) {
        alluxio.shaded.client.com.letionService.take().get();
      }
    } catch (ExecutionException e) {
      LOG.error("Parallel decompress rocksdb fail", e);
      FileUtils.deletePathRecursively(dirPath.toString());
      throw new IOException(e);
    } catch (InterruptedException e) {
      LOG.info("Parallel decompress rocksdb interrupted");
      FileUtils.deletePathRecursively(dirPath.toString());
      Thread.currentThread().interrupt();
      throw new RuntimeException(e);
    } finally {
      ExecutorServiceUtils.shutdownAndAwaitTermination(executor);
    }
  }

  /**
   * Unzip entry in ZipFile.
   *
   * @param zipFile
   * @param entry
   * @param dirPath
   * @throws Exception
   */
  private static void unzipEntry(ZipFile zipFile, ZipArchiveEntry entry, Path dirPath)
      throws Exception {
    File outputFile = new File(dirPath.toFile(), entry.getName());
    outputFile.getParentFile().mkdirs();

    if (entry.isDirectory()) {
      outputFile.mkdir();
    } else {
      try (InputStream inputStream = zipFile.getInputStream(entry);
           FileOutputStream fileOutputStream = new FileOutputStream(outputFile)) {
        IOUtils.copy(inputStream, fileOutputStream);
      }
    }
  }

  private ParallelZipUtils() {} // Utils class
}
