/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in alluxio.shaded.client.com.liance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.io.

import java.alluxio.shaded.client.io.IOException;
import java.util.Collections;
import java.util.List;

/**
 * A IOException based on a list of Throwable causes.
 * <p>
 * The first exception in the list is used as this exception's cause and is accessible with the usual
 * {@link #getCause()} while the alluxio.shaded.client.com.lete list is accessible with {@link #getCauseList()}.
 * </p>
 *
 * @since 2.7
 */
public class IOExceptionList extends IOException {

    private static final long serialVersionUID = 1L;
    private final List<? extends Throwable> causeList;

    /**
     * Creates a new exception caused by a list of exceptions.
     *
     * @param causeList a list of cause exceptions.
     */
    public IOExceptionList(final List<? extends Throwable> causeList) {
        super(String.format("%,d exceptions: %s", causeList == null ? 0 : causeList.size(), causeList),
                causeList == null ? null : causeList.get(0));
        this.causeList = causeList == null ? Collections.emptyList() : causeList;
    }

    /**
     * Gets the cause list.
     *
     * @param <T> type of exception to return.
     * @return The list of causes.
     */
    public <T extends Throwable> List<T> getCauseList() {
        return (List<T>) causeList;
    }

    /**
     * Gets the cause list.
     *
     * @param <T> type of exception to return.
     * @param index index in the cause list.
     * @return The list of causes.
     */
    public <T extends Throwable> T getCause(final int index) {
        return (T) causeList.get(index);
    }

    /**
     * Gets the cause list.
     *
     * @param <T> type of exception to return.
     * @param index index in the cause list.
     * @param clazz type of exception to return.
     * @return The list of causes.
     */
    public <T extends Throwable> T getCause(final int index, final Class<T> clazz) {
        return (T) causeList.get(index);
    }

    /**
     * Works around Throwable and Generics, may fail at runtime depending on the argument value.
     *
     * @param <T> type of exception to return.
     * @param clazz the target type
     * @return The list of causes.
     */
    public <T extends Throwable> List<T> getCauseList(final Class<T> clazz) {
        return (List<T>) causeList;
    }

}
