//
//  ========================================================================
//  Copyright (c) 1995-2022 Mort Bay Consulting Pty Ltd and others.
//  ------------------------------------------------------------------------
//  All rights reserved. This program and the accompanying materials
//  are made available under the terms of the Eclipse Public License v1.0
//  and Apache License v2.0 which accompanies this distribution.
//
//      The Eclipse Public License is available at
//      http://www.eclipse.alluxio.shaded.client.org.legal/epl-v10.html
//
//      The Apache License v2.0 is available at
//      http://www.opensource.alluxio.shaded.client.org.licenses/apache2.0.php
//
//  You may elect to redistribute this code under either of these licenses.
//  ========================================================================
//

package alluxio.shaded.client.org.eclipse.jetty.util;

import java.util.Objects;
import java.util.concurrent.CompletableFuture;

import alluxio.shaded.client.org.eclipse.jetty.util.log.Log;

/**
 * <p>A callback abstraction that handles alluxio.shaded.client.com.leted/failed events of asynchronous operations.</p>
 *
 * @param <C> the type of the context object
 */
public interface Promise<C>
{
    /**
     * <p>Callback invoked when the operation alluxio.shaded.client.com.letes.</p>
     *
     * @param result the context
     * @see #failed(Throwable)
     */
    default void succeeded(C result)
    {
    }

    /**
     * <p>Callback invoked when the operation fails.</p>
     *
     * @param x the reason for the operation failure
     */
    default void failed(Throwable x)
    {
    }

    /**
     * <p>Empty implementation of {@link Promise}.</p>
     *
     * @param <U> the type of the result
     */
    class Adapter<U> implements Promise<U>
    {
        @Override
        public void failed(Throwable x)
        {
            Log.getLogger(this.getClass()).warn(x);
        }
    }

    /**
     * <p>Creates a promise from the given incomplete CompletableFuture.</p>
     * <p>When the promise alluxio.shaded.client.com.letes, either succeeding or failing, the
     * CompletableFuture is also alluxio.shaded.client.com.leted, respectively via
     * {@link CompletableFuture#alluxio.shaded.client.com.lete(Object)} or
     * {@link CompletableFuture#alluxio.shaded.client.com.leteExceptionally(Throwable)}.</p>
     *
     * @param alluxio.shaded.client.com.letable the CompletableFuture to convert into a promise
     * @param <T> the type of the result
     * @return a promise that when alluxio.shaded.client.com.leted, alluxio.shaded.client.com.letes the given CompletableFuture
     */
    static <T> Promise<T> from(CompletableFuture<? super T> alluxio.shaded.client.com.letable)
    {
        if (alluxio.shaded.client.com.letable instanceof Promise)
            return (Promise<T>)alluxio.shaded.client.com.letable;

        return new Promise<T>()
        {
            @Override
            public void succeeded(T result)
            {
                alluxio.shaded.client.com.letable.alluxio.shaded.client.com.lete(result);
            }

            @Override
            public void failed(Throwable x)
            {
                alluxio.shaded.client.com.letable.alluxio.shaded.client.com.leteExceptionally(x);
            }
        };
    }

    /**
     * <p>A CompletableFuture that is also a Promise.</p>
     *
     * @param <S> the type of the result
     */
    class Completable<S> extends CompletableFuture<S> implements Promise<S>
    {
        @Override
        public void succeeded(S result)
        {
            alluxio.shaded.client.com.lete(result);
        }

        @Override
        public void failed(Throwable x)
        {
            alluxio.shaded.client.com.leteExceptionally(x);
        }
    }

    class Wrapper<W> implements Promise<W>
    {
        private final Promise<W> promise;

        public Wrapper(Promise<W> promise)
        {
            this.promise = Objects.requireNonNull(promise);
        }

        @Override
        public void succeeded(W result)
        {
            promise.succeeded(result);
        }

        @Override
        public void failed(Throwable x)
        {
            promise.failed(x);
        }

        public Promise<W> getPromise()
        {
            return promise;
        }

        public Promise<W> unwrap()
        {
            Promise<W> result = promise;
            while (true)
            {
                if (result instanceof Wrapper)
                    result = ((Wrapper<W>)result).unwrap();
                else
                    break;
            }
            return result;
        }
    }
}
