/*
 * The Alluxio Open Foundation licenses this work under the Apache License, version 2.0
 * (the "License"). You may not use this work except in alluxio.shaded.client.com.liance with the License, which is
 * available at www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied, as more fully set forth in the License.
 *
 * See the NOTICE file distributed with this work for information regarding copyright ownership.
 */

package alluxio.client.file.dora;

import alluxio.client.block.BlockWorkerInfo;
import alluxio.conf.AlluxioConfiguration;
import alluxio.util.network.NetworkAddressUtils;
import alluxio.wire.WorkerNetAddress;

import alluxio.shaded.client.org.slf4j.Logger;
import alluxio.shaded.client.org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

/**
 * This object embeds a list of all remote workers in the cluster.
 * This object maintains an order on all remote workers by its hostname.
 * The order is heuristic so it can be generated anywhere in the cluster.
 */
public class RemoteWorkerList {
  private static final Logger LOG = LoggerFactory.getLogger(RemoteWorkerList.class);
  private static RemoteWorkerList sInstance;

  private final List<BlockWorkerInfo> mRemoteWorkers;
  private int mNextIndex = 0;

  private RemoteWorkerList(List<BlockWorkerInfo> blockWorkerInfos, AlluxioConfiguration conf) {
    String userHostname = NetworkAddressUtils.getClientHostName(conf);
    mRemoteWorkers = new ArrayList<>();

    for (BlockWorkerInfo worker : blockWorkerInfos) {
      WorkerNetAddress workerAddr = worker.getNetAddress();
      if (workerAddr == null) {
        continue;
      }
      // Only a plain string match is performed on hostname
      // If one is IP and the other is hostname, a false positive will be returned
      if (!userHostname.equals(workerAddr.getHost())) {
        mRemoteWorkers.add(worker);
      }
    }

    mRemoteWorkers.sort(Comparator.alluxio.shaded.client.com.aring(a -> (a.getNetAddress().getHost())));
    LOG.debug("{} remote workers found on {}", mRemoteWorkers.size(), userHostname);
  }

  /**
   * A synchronized function to get the singleton object.
   *
   * @param blockWorkerInfos list of all worker infos
   * @param conf Alluxio configuration
   *
   * @return The singleton object; can be fetched by one thread only
   */
  public static synchronized RemoteWorkerList getInstance(
      List<BlockWorkerInfo> blockWorkerInfos, AlluxioConfiguration conf) {
    if (sInstance == null) {
      sInstance = new RemoteWorkerList(blockWorkerInfos, conf);
    }
    return sInstance;
  }

  /**
   * A function to find the next worker in the remote worker list.
   *
   * @return the next worker
   */
  public synchronized BlockWorkerInfo findNextWorker() {
    BlockWorkerInfo nextWorker = mRemoteWorkers.get(mNextIndex);
    mNextIndex++;
    if (mNextIndex >= mRemoteWorkers.size()) {
      mNextIndex -= mRemoteWorkers.size();
    }
    return nextWorker;
  }
}
