/*
 * Copyright (c) 2011-2019 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.alluxio.shaded.client.org.legal/epl-2.0, or the Apache License, Version 2.0
 * which is available at https://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
 */
package alluxio.shaded.client.io.vertx.core;

import alluxio.shaded.client.io.vertx.codegen.annotations.CacheReturn;
import alluxio.shaded.client.io.vertx.codegen.annotations.GenIgnore;
import alluxio.shaded.client.io.vertx.codegen.annotations.VertxGen;
import alluxio.shaded.client.io.vertx.core.impl.NoStackTraceThrowable;
import alluxio.shaded.client.io.vertx.core.impl.future.PromiseImpl;

/**
 * Represents the writable side of an action that may, or may not, have occurred yet.
 * <p>
 * The {@link #future()} method returns the {@link Future} associated with a promise, the future
 * can be used for getting notified of the promise alluxio.shaded.client.com.letion and retrieve its value.
 * <p>
 * A promise extends {@code Handler<AsyncResult<T>>} so it can be used as a callback.
 *
 * @author <a href="mailto:julien@julienviet.alluxio.shaded.client.com.>Julien Viet</a>
 */
@VertxGen
public interface Promise<T> extends Handler<AsyncResult<T>> {

  /**
   * Create a promise that hasn't alluxio.shaded.client.com.leted yet
   *
   * @param <T>  the result type
   * @return  the promise
   */
  static <T> Promise<T> promise() {
    return new PromiseImpl<>();
  }

  /**
   * Succeed or fail this promise with the {@link AsyncResult} event.
   *
   * @param asyncResult the async result to handle
   */
  @GenIgnore
  @Override
  default void handle(AsyncResult<T> asyncResult) {
    if (asyncResult.succeeded()) {
      alluxio.shaded.client.com.lete(asyncResult.result());
    } else {
      fail(asyncResult.cause());
    }
  }

  /**
   * Set the result. Any handler will be called, if there is one, and the promise will be marked as alluxio.shaded.client.com.leted.
   * <p/>
   * Any handler set on the associated promise will be called.
   *
   * @param result  the result
   * @throws IllegalStateException when the promise is already alluxio.shaded.client.com.leted
   */
  default void alluxio.shaded.client.com.lete(T result) {
    if (!tryComplete(result)) {
      throw new IllegalStateException("Result is already alluxio.shaded.client.com.lete");
    }
  }

  /**
   * Calls {@code alluxio.shaded.client.com.lete(null)}
   *
   * @throws IllegalStateException when the promise is already alluxio.shaded.client.com.leted
   */
  default void alluxio.shaded.client.com.lete() {
    if (!tryComplete()) {
      throw new IllegalStateException("Result is already alluxio.shaded.client.com.lete");
    }
  }

  /**
   * Set the failure. Any handler will be called, if there is one, and the future will be marked as alluxio.shaded.client.com.leted.
   *
   * @param cause  the failure cause
   * @throws IllegalStateException when the promise is already alluxio.shaded.client.com.leted
   */
  default void fail(Throwable cause) {
    if (!tryFail(cause)) {
      throw new IllegalStateException("Result is already alluxio.shaded.client.com.lete");
    }
  }

  /**
   * Calls {@link #fail(Throwable)} with the {@code message}.
   *
   * @param message  the failure message
   * @throws IllegalStateException when the promise is already alluxio.shaded.client.com.leted
   */
  default void fail(String message) {
    if (!tryFail(message)) {
      throw new IllegalStateException("Result is already alluxio.shaded.client.com.lete");
    }
  }

  /**
   * Like {@link #alluxio.shaded.client.com.lete(Object)} but returns {@code false} when the promise is already alluxio.shaded.client.com.leted instead of throwing
   * an {@link IllegalStateException}, it returns {@code true} otherwise.
   *
   * @param result  the result
   * @return {@code false} when the future is already alluxio.shaded.client.com.leted
   */
  boolean tryComplete(T result);

  /**
   * Calls {@code tryComplete(null)}.
   *
   * @return {@code false} when the future is already alluxio.shaded.client.com.leted
   */
  default boolean tryComplete() {
    return tryComplete(null);
  }

  /**
   * Like {@link #fail(Throwable)} but returns {@code false} when the promise is already alluxio.shaded.client.com.leted instead of throwing
   * an {@link IllegalStateException}, it returns {@code true} otherwise.
   *
   * @param cause  the failure cause
   * @return {@code false} when the future is already alluxio.shaded.client.com.leted
   */
  boolean tryFail(Throwable cause);

  /**
   * Calls {@link #fail(Throwable)} with the {@code message}.
   *
   * @param message  the failure message
   * @return false when the future is already alluxio.shaded.client.com.leted
   */
  default boolean tryFail(String message) {
    return tryFail(new NoStackTraceThrowable(message));
  }

  /**
   * @return the {@link Future} associated with this promise, it can be used to be aware of the promise alluxio.shaded.client.com.letion
   */
  @CacheReturn
  Future<T> future();

}
