/*
 * Copyright 2016-2021 The jetcd authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.io.etcd.jetcd.impl;

import java.util.concurrent.CompletableFuture;

import alluxio.shaded.client.io.etcd.jetcd.ByteSequence;
import alluxio.shaded.client.io.etcd.jetcd.Lock;
import alluxio.shaded.client.io.etcd.jetcd.api.lock.LockRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.lock.UnlockRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.lock.VertxLockGrpc;
import alluxio.shaded.client.io.etcd.jetcd.lock.LockResponse;
import alluxio.shaded.client.io.etcd.jetcd.lock.UnlockResponse;
import alluxio.shaded.client.io.etcd.jetcd.support.Errors;
import alluxio.shaded.client.io.etcd.jetcd.support.Util;

import static alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.base.Preconditions.checkNotNull;

final class LockImpl extends Impl implements Lock {
    private final VertxLockGrpc.LockVertxStub stub;
    private final ByteSequence namespace;

    LockImpl(ClientConnectionManager connectionManager) {
        super(connectionManager);

        this.stub = connectionManager.newStub(VertxLockGrpc::newVertxStub);
        this.namespace = connectionManager.getNamespace();
    }

    @Override
    public CompletableFuture<LockResponse> lock(ByteSequence name, long leaseId) {
        checkNotNull(name);

        LockRequest request = LockRequest.newBuilder()
            .setName(Util.prefixNamespace(name, namespace))
            .setLease(leaseId)
            .build();

        return execute(
            () -> stub.lock(request),
            response -> new LockResponse(response, namespace),
            Errors::isRetryable);
    }

    @Override
    public CompletableFuture<UnlockResponse> unlock(ByteSequence lockKey) {
        checkNotNull(lockKey);

        UnlockRequest request = UnlockRequest.newBuilder()
            .setKey(Util.prefixNamespace(lockKey, namespace))
            .build();

        return execute(
            () -> stub.unlock(request),
            UnlockResponse::new,
            Errors::isRetryable);
    }
}
