/*
 * Copyright (c) 2011-2019 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.alluxio.shaded.client.org.legal/epl-2.0, or the Apache License, Version 2.0
 * which is available at https://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
 */

package alluxio.shaded.client.io.vertx.core.net.impl.transport;

import alluxio.shaded.client.io.netty.bootstrap.ServerBootstrap;
import alluxio.shaded.client.io.netty.channel.Channel;
import alluxio.shaded.client.io.netty.channel.ChannelFactory;
import alluxio.shaded.client.io.netty.channel.EventLoopGroup;
import alluxio.shaded.client.io.netty.channel.ServerChannel;
import alluxio.shaded.client.io.netty.channel.kqueue.*;
import alluxio.shaded.client.io.netty.channel.socket.DatagramChannel;
import alluxio.shaded.client.io.netty.channel.socket.InternetProtocolFamily;
import alluxio.shaded.client.io.netty.channel.unix.DomainSocketAddress;
import alluxio.shaded.client.io.vertx.core.datagram.DatagramSocketOptions;
import alluxio.shaded.client.io.vertx.core.net.NetServerOptions;
import alluxio.shaded.client.io.vertx.core.net.impl.SocketAddressImpl;

import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.util.concurrent.ThreadFactory;

/**
 * @author <a href="mailto:julien@julienviet.alluxio.shaded.client.com.>Julien Viet</a>
 */
class KQueueTransport extends Transport {

  KQueueTransport() {
  }

  @Override
  public SocketAddress convert(alluxio.shaded.client.io.vertx.core.net.SocketAddress address) {
    if (address.isDomainSocket()) {
      return new DomainSocketAddress(address.path());
    } else {
      return super.convert(address);
    }
  }

  @Override
  public alluxio.shaded.client.io.vertx.core.net.SocketAddress convert(SocketAddress address) {
    if (address instanceof DomainSocketAddress) {
      return new SocketAddressImpl(((DomainSocketAddress) address).path());
    }
    return super.convert(address);
  }

  @Override
  public boolean isAvailable() {
    return KQueue.isAvailable();
  }

  @Override
  public Throwable unavailabilityCause() {
    return KQueue.unavailabilityCause();
  }

  @Override
  public EventLoopGroup eventLoopGroup(int type, int nThreads, ThreadFactory threadFactory, int alluxio.shaded.client.io.atio) {
    KQueueEventLoopGroup eventLoopGroup = new KQueueEventLoopGroup(nThreads, threadFactory);
    eventLoopGroup.setIoRatio(alluxio.shaded.client.io.atio);
    return eventLoopGroup;
  }

  @Override
  public DatagramChannel datagramChannel() {
    return new KQueueDatagramChannel();
  }

  @Override
  public DatagramChannel datagramChannel(InternetProtocolFamily family) {
    return new KQueueDatagramChannel();
  }

  @Override
  public ChannelFactory<? extends Channel> channelFactory(boolean domainSocket) {
    if (domainSocket) {
      return KQueueDomainSocketChannel::new;
    } else {
      return KQueueSocketChannel::new;
    }
  }

  @Override
  public ChannelFactory<? extends ServerChannel> serverChannelFactory(boolean domainSocket) {
    if (domainSocket) {
      return KQueueServerDomainSocketChannel::new;
    } else {
      return KQueueServerSocketChannel::new;
    }
  }

  @Override
  public void configure(NetServerOptions options, boolean domainSocket, ServerBootstrap bootstrap) {
    if (!domainSocket) {
      bootstrap.option(KQueueChannelOption.SO_REUSEPORT, options.isReusePort());
    }
    super.configure(options, domainSocket, bootstrap);
  }

  @Override
  public void configure(DatagramChannel channel, DatagramSocketOptions options) {
    channel.config().setOption(KQueueChannelOption.SO_REUSEPORT, options.isReusePort());
    super.configure(channel, options);
  }
}
