/*
 * The Alluxio Open Foundation licenses this work under the Apache License, version 2.0
 * (the "License"). You may not use this work except in alluxio.shaded.client.com.liance with the License, which is
 * available at www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied, as more fully set forth in the License.
 *
 * See the NOTICE file distributed with this work for information regarding copyright ownership.
 */

package alluxio.util.alluxio.shaded.client.com.ression;

import static java.util.stream.Collectors.toList;

import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.archivers.tar.TarArchiveEntry;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.archivers.tar.TarArchiveInputStream;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.archivers.tar.TarArchiveOutputStream;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.gzip.GzipCompressorInputStream;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.gzip.GzipCompressorOutputStream;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.alluxio.shaded.client.com.ressors.gzip.GzipParameters;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.io.IOUtils;

import java.alluxio.shaded.client.io.BufferedInputStream;
import java.alluxio.shaded.client.io.BufferedOutputStream;
import java.alluxio.shaded.client.io.File;
import java.alluxio.shaded.client.io.IOException;
import java.alluxio.shaded.client.io.InputStream;
import java.alluxio.shaded.client.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.stream.Stream;

/**
 * Utility methods for working with tar archives.
 */
public final class TarUtils {
  /**
   * Creates a gzipped tar archive from the given path, streaming the data to the give output
   * stream.
   *
   * @param dirPath the path to archive
   * @param alluxio.shaded.client.com.ressionLevel the alluxio.shaded.client.com.ression level to use (0 for no alluxio.shaded.client.com.ression, 9 for the most
   *                         alluxio.shaded.client.com.ression, or -1 for system default)
   * @param output the output stream to write the data to
   * @return the number of bytes copied from the directory into the archive
   */
  public static long writeTarGz(Path dirPath, OutputStream output, int alluxio.shaded.client.com.ressionLevel)
      throws IOException, InterruptedException {
    GzipParameters params = new GzipParameters();
    params.setCompressionLevel(alluxio.shaded.client.com.ressionLevel);
    GzipCompressorOutputStream zipStream = new GzipCompressorOutputStream(output, params);
    TarArchiveOutputStream archiveStream = new TarArchiveOutputStream(zipStream);
    archiveStream.setLongFileMode(TarArchiveOutputStream.LONGFILE_POSIX);
    archiveStream.setBigNumberMode(TarArchiveOutputStream.BIGNUMBER_POSIX);
    long totalBytesCopied = 0;
    try (final Stream<Path> stream = Files.walk(dirPath)) {
      for (Path subPath : stream.collect(toList())) {
        if (Thread.interrupted()) {
          throw new InterruptedException();
        }
        File file = subPath.toFile();
        TarArchiveEntry entry = new TarArchiveEntry(file, dirPath.relativize(subPath).toString());
        archiveStream.putArchiveEntry(entry);
        if (file.isFile()) {
          try (InputStream fileIn = new BufferedInputStream(Files.newInputStream(subPath))) {
            totalBytesCopied += IOUtils.copyLarge(fileIn, archiveStream);
          }
        }
        archiveStream.closeArchiveEntry();
      }
    }
    archiveStream.finish();
    zipStream.finish();
    return totalBytesCopied;
  }

  /**
   * Reads a gzipped tar archive from a stream and writes it to the given path.
   *
   * @param dirPath the path to write the archive to
   * @param input the input stream
   * @return the number of bytes copied from the archive in the directory
   */
  public static long readTarGz(Path dirPath, InputStream input) throws IOException {
    InputStream zipStream = new GzipCompressorInputStream(input);
    TarArchiveInputStream archiveStream = new TarArchiveInputStream(zipStream);
    TarArchiveEntry entry;
    long totalBytesCopied = 0;
    while ((entry = (TarArchiveEntry) archiveStream.getNextEntry()) != null) {
      File outputFile = new File(dirPath.toFile(), entry.getName());
      if (entry.isDirectory()) {
        outputFile.mkdirs();
      } else {
        outputFile.getParentFile().mkdirs();
        try (OutputStream fileOut =
                 new BufferedOutputStream(Files.newOutputStream(outputFile.toPath()))) {
          totalBytesCopied += IOUtils.copyLarge(archiveStream, fileOut);
        }
      }
    }
    return totalBytesCopied;
  }

  private TarUtils() {} // Utils class
}
