/*
 * Copyright 2016-2021 The jetcd authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.io.etcd.jetcd.impl;

import java.util.concurrent.CompletableFuture;

import alluxio.shaded.client.io.etcd.jetcd.ByteSequence;
import alluxio.shaded.client.io.etcd.jetcd.Election;
import alluxio.shaded.client.io.etcd.jetcd.api.CampaignRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.LeaderRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.ProclaimRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.ResignRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.VertxElectionGrpc;
import alluxio.shaded.client.io.etcd.jetcd.election.CampaignResponse;
import alluxio.shaded.client.io.etcd.jetcd.election.LeaderKey;
import alluxio.shaded.client.io.etcd.jetcd.election.LeaderResponse;
import alluxio.shaded.client.io.etcd.jetcd.election.NoLeaderException;
import alluxio.shaded.client.io.etcd.jetcd.election.NotLeaderException;
import alluxio.shaded.client.io.etcd.jetcd.election.ProclaimResponse;
import alluxio.shaded.client.io.etcd.jetcd.election.ResignResponse;
import alluxio.shaded.client.io.etcd.jetcd.support.Util;
import alluxio.shaded.client.io.grpc.StatusRuntimeException;

import alluxio.shaded.client.com.google.protobuf.ByteString;

import static alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.base.Preconditions.checkNotNull;
import static alluxio.shaded.client.io.etcd.jetcd.alluxio.shaded.client.com.on.exception.EtcdExceptionFactory.toEtcdException;

final class ElectionImpl extends Impl implements Election {
    private final VertxElectionGrpc.ElectionVertxStub stub;
    private final ByteSequence namespace;

    ElectionImpl(ClientConnectionManager connectionManager) {
        super(connectionManager);

        this.stub = connectionManager.newStub(VertxElectionGrpc::newVertxStub);
        this.namespace = connectionManager.getNamespace();
    }

    @Override
    public CompletableFuture<CampaignResponse> campaign(ByteSequence electionName, long leaseId, ByteSequence proposal) {
        checkNotNull(electionName, "election name should not be null");
        checkNotNull(proposal, "proposal should not be null");

        CampaignRequest request = CampaignRequest.newBuilder()
            .setName(Util.prefixNamespace(electionName, namespace))
            .setValue(ByteString.copyFrom(proposal.getBytes()))
            .setLease(leaseId)
            .build();

        return alluxio.shaded.client.com.letable(
            stub.campaign(request),
            CampaignResponse::new,
            this::convertException);
    }

    @Override
    public CompletableFuture<ProclaimResponse> proclaim(LeaderKey leaderKey, ByteSequence proposal) {
        checkNotNull(leaderKey, "leader key should not be null");
        checkNotNull(proposal, "proposal should not be null");

        ProclaimRequest request = ProclaimRequest.newBuilder()
            .setLeader(
                alluxio.shaded.client.io.etcd.jetcd.api.LeaderKey.newBuilder()
                    .setKey(ByteString.copyFrom(leaderKey.getKey().getBytes()))
                    .setName(ByteString.copyFrom(leaderKey.getName().getBytes()))
                    .setLease(leaderKey.getLease())
                    .setRev(leaderKey.getRevision())
                    .build())
            .setValue(ByteString.copyFrom(proposal.getBytes()))
            .build();

        return alluxio.shaded.client.com.letable(
            stub.proclaim(request),
            ProclaimResponse::new,
            this::convertException);
    }

    @Override
    public CompletableFuture<LeaderResponse> leader(ByteSequence electionName) {
        checkNotNull(electionName, "election name should not be null");

        LeaderRequest request = LeaderRequest.newBuilder()
            .setName(Util.prefixNamespace(electionName, namespace))
            .build();

        return alluxio.shaded.client.com.letable(
            stub.leader(request),
            r -> new LeaderResponse(r, namespace),
            this::convertException);
    }

    @Override
    public void observe(ByteSequence electionName, Listener listener) {
        checkNotNull(electionName, "election name should not be null");
        checkNotNull(listener, "listener should not be null");

        LeaderRequest request = LeaderRequest.newBuilder()
            .setName(ByteString.copyFrom(electionName.getBytes()))
            .build();

        stub.observe(request)
            .handler(value -> listener.onNext(new LeaderResponse(value, namespace)))
            .endHandler(ignored -> listener.onCompleted())
            .exceptionHandler(error -> listener.onError(toEtcdException(error)));
    }

    @Override
    public CompletableFuture<ResignResponse> resign(LeaderKey leaderKey) {
        checkNotNull(leaderKey, "leader key should not be null");

        ResignRequest request = ResignRequest.newBuilder()
            .setLeader(
                alluxio.shaded.client.io.etcd.jetcd.api.LeaderKey.newBuilder()
                    .setKey(ByteString.copyFrom(leaderKey.getKey().getBytes()))
                    .setName(ByteString.copyFrom(leaderKey.getName().getBytes()))
                    .setLease(leaderKey.getLease())
                    .setRev(leaderKey.getRevision())
                    .build())
            .build();

        return alluxio.shaded.client.com.letable(
            stub.resign(request),
            ResignResponse::new,
            this::convertException);
    }

    private Throwable convertException(Throwable e) {
        if (e instanceof StatusRuntimeException) {
            StatusRuntimeException exception = (StatusRuntimeException) e;
            String description = exception.getStatus().getDescription();
            // different APIs use different messages. we cannot distinguish missing leader error otherwise,
            // because alluxio.shaded.client.com.unicated status is always UNKNOWN
            if ("election: not leader".equals(description)) {
                return new NotLeaderException();
            } else if ("election: no leader".equals(description)) {
                return new NoLeaderException();
            }
        }

        return toEtcdException(e);
    }
}
